#!/bin/sh

# ==== definition ====
SCRIPT_VERSION="1.6.1"

exit_hook() {
  if [ -f /etc/apt/sources.list.bak.tmp.ghjy662 ]; then
    # recover original source.list
    mv /etc/apt/sources.list.bak.tmp.ghjy662 /etc/apt/sources.list
    echo "[INFO] Recovered original source.list."
  fi
  trap - EXIT  # Remove the trap for EXIT to prevent recursion
  exit 1
}

check_deb() {
  PKGNAME="$1"
  FILE="$2"

  # check if deb file exists
  if [ ! -f "${FILE}" ]; then
    echo "[ERROR] File not found: "${FILE}"" >&2
    exit 1
  fi

  DEBCONTENT="$(dpkg-deb -I "${FILE}")"
  if [ -z "$(echo \"${DEBCONTENT}\" | grep 'Package:' | grep "${PKGNAME}")" ]; then
    echo "[ERROR] Invalid deb file: "${FILE}"" >&2
    exit 1
  fi
}

find_filepath_by_pkg() {
  CHANNEL="/usr/share/mxview-one-deps/dists/$(lsb_release -cs)/main/binary-amd64"
  PROFILE="${CHANNEL}/Packages.gz"
  PKGNAME="$1"
  STR="/^Package: $PKGNAME$/,/^$/ { if (/Filename:/) print NR }"
  NR=$(zcat "$PROFILE" | awk "$STR")
  if [ -z "${NR}" ]; then
    echo "[ERROR] Can't find package: ${PKGNAME}" >&2
    exit 1
  fi
  FPATH=$(zcat "$PROFILE" | sed -n "${NR}p" | awk -F'Filename: ' '{print $2}')
  FULLPATH="${CHANNEL}/${FPATH}"
  echo "$(basename "${FULLPATH}")"
}

install_pkg() {
  CHANNEL="/usr/share/mxview-one-deps/dists/$(lsb_release -cs)/main/binary-amd64"
  PKGNAME="$1"
  VERSION="$(find_filepath_by_pkg "$1")"
  if [ -z "${VERSION}" ]; then
    echo "[ERROR] Can't find package: ${PKGNAME}"
    exit 1
  fi
  echo "[HINT] Found package in mxview-one-deps channel: ${PKGNAME}=${VERSION}"
  echo "[HINT] However, if your system has newer version in other registered apt channel, the newer one will be installed from other apt channel."
  echo "[HINT] If you still want to install the version we collected for you, you need to remove the other apt channel first."
  apt install --no-install-recommends -y "${PKGNAME}"
  echo "[INFO] Package installed: ${PKGNAME}"
}

dpkg_install() {
  CHANNEL="/usr/share/mxview-one-deps/dists/$(lsb_release -cs)/main/binary-amd64"
  PKGNAME="$1"
  VERSION="$(find_filepath_by_pkg "$1")"
  if [ -z "${VERSION}" ]; then
    echo "[ERROR] Can't find package: ${PKGNAME}"
    exit 1
  fi
  echo "[HINT] Found package in mxview-one-deps channel: ${PKGNAME}=${VERSION} using dpkg -i"
  dpkg --ignore-depends=libc6 -i "${CHANNEL}/${VERSION}"
  echo "[INFO] Package installed: ${PKGNAME}"
}


# ==== process ====

echo "[INFO] Start mxview-one ubuntu installer script: v${SCRIPT_VERSION}"
echo "[INFO] Supported ubuntu version: bionic, focal, jammy"

CODENAME="$(lsb_release -cs)"
if [ "${CODENAME}" != "bionic" ] && [ "${CODENAME}" != "focal" ] && [ "${CODENAME}" != "jammy" ]; then
  echo "[ERROR] Unsupported ubuntu version: ${CODENAME}" >&2
  exit 1
fi

if [ "root" != "$(whoami)" ]; then
  echo "[ERROR] Please run as root or use sudo" >&2
  exit 1
fi

if [ "amd64" != "$(dpkg --print-architecture)" ]; then
  echo "[ERROR] This dependency pack supports amd64 CPU arch only." >&2
  exit 1
fi

# valid the input deb
DEBFILE="$1"
check_deb "mxview" "${DEBFILE}" || exit 1

# set up parameters
OFFLINE_DEP="${OFFLINE_DEP:=}"

# check parameters
echo "[INFO] OFFLINE_DEP=${OFFLINE_DEP}"

# stop dependent services
systemctl stop postgresql

# temporary disable existing source.list
trap exit_hook INT TERM EXIT
if [ -f /etc/apt/sources.list ]; then
  mv /etc/apt/sources.list /etc/apt/sources.list.bak.tmp.ghjy662
  echo "[INFO] Temporarily disabled original source.list by moving it to /etc/apt/sources.list.bak.tmp.ghjy662"
fi
apt update -y

# install dependent packages
if [ ! -s "${OFFLINE_DEP}" ]; then
  echo "[ERROR] Missing offline dependent packages" >&2
  exit 1
fi
DEPNAME="mxview-one-deps"
echo "[INFO] Installing offline dependent packages from: ${OFFLINE_DEP}"
check_deb "${DEPNAME}" "${OFFLINE_DEP}" || exit 1
dpkg -i "${OFFLINE_DEP}"

install_pkg "postgresql-15" || exit 1
install_pkg "nodejs" || exit 1

dpkg_install "libc-dev-bin" || exit 1
dpkg_install "linux-libc-dev" || exit 1
if [ "${CODENAME}" != "bionic" ]; then
  dpkg_install "libcrypt-dev" || exit 1
fi
if [ "${CODENAME}" = "jammy" ]; then
  dpkg_install "rpcsvc-proto" || exit 1
  dpkg_install "libtirpc-dev" || exit 1
  dpkg_install "libnsl-dev" || exit 1
fi
dpkg_install "libc6-dev" || exit 1
if [ "${CODENAME}" = "jammy" ]; then
  dpkg_install "libdpkg-perl" || exit 1
  dpkg_install "pkg-config" || exit 1
  dpkg_install "libdbus-1-dev" || exit 1
fi
dpkg_install "libpcap0.8" || exit 1
dpkg_install "libpcap0.8-dev" || exit 1
dpkg_install "libpcap-dev" || exit 1

echo "[INFO] Offline dependent packages installed."

# stop dependent services
systemctl stop postgresql

# install mxview one
EXIT_CODE=0
DEPENDS="$(dpkg-deb -I "${DEBFILE}" | grep Depends: | grep postgresql-15)"
if [ -n "$DEPENDS" ]; then
  echo "[INFO] Found new .deb."
  dpkg -i "${DEBFILE}"
else
  echo "[INFO] Found legacy .deb."
  sudo dpkg --ignore-depends=postgresql -i "${DEBFILE}"

  # recover apt system
  APTCFG=/var/lib/dpkg/status
  CHECKER="$(awk '/mxview/,/^$/ { if (/Depends:/) print }' ${APTCFG} | grep postgresql-15)"
  if [ -n "${CHECKER}" ]; then
    echo "[INFO] mxview-one dpkg info already patched."
  else
    NR=$(awk '/mxview/,/^$/ { if (/Depends:/) print NR }' ${APTCFG})
    if [ -z "$NR" ]; then
      echo "[ERROR] Can't find mxview pkg in ${APTCFG}"
      EXIT_CODE=1
    fi
    echo "[INFO] Extracted NR=$NR"
    echo "[INFO] Patching mxview dpkg info..."
    EXP="${NR}s/postgresql (>= 10)/postgresql (>= 10) | postgresql-15/"
    sed -i "$EXP" "${APTCFG}"
    echo "[INFO] mxview-one dpkg info patched."
  fi
fi

# recover original source.list
mv /etc/apt/sources.list.bak.tmp.ghjy662 /etc/apt/sources.list
echo "[INFO] Recovered original source.list."

# remove dependent packages
dpkg -r "${DEPNAME}"
dpkg --purge "${DEPNAME}"
echo "[INFO] Removed offline dependent packages."

# Complete
if [ ${EXIT_CODE} -ne 0 ]; then
  echo "[INFO] Failed to install mxview-one."
  exit ${EXIT_CODE}
else
  echo "[INFO] Successfully installed mxview-one."
  exit 0
fi
