/**
 * @file mcm-base-info.h
 *
 * This library is for Moxa Connection Management
 *
 * @author York LY Chang <YorkLY.Chang@moxa.com>
 */

#ifndef _MCM_BASE_INFO_H_
#define _MCM_BASE_INFO_H_

#include <inttypes.h>
#include <stdbool.h>
/**
 *  MCM Error Number
 */
enum MCM_ERROR_CODE {
    MCM_SUCCESS     = 0, /**< success */
    MCM_GENERIC_ERR = 1, /**< Generic error */
    MCM_ENOMEM      = 2, /**< Allocate memory fail*/
    MCM_ENODEV      = 3, /**< No such device */
    MCM_EINVAL      = 4, /**< Invalid argument */
};

/**
 *  mcm_object_info
 */
struct mcm_object_info {
    char                   *interface_name; /**< name */
    char                   *device_name;    /**< device name */
    char                   *device_type;    /**< device type */
    struct mcm_object_info *next;           /**< linked-list */
};

typedef struct mcm_object_info mcm_object_info;

/**
 *  mcm_property_info_linked_list
 */
struct mcm_property_info {
    char *propertyName;  /**< property name */
    char *propertyValue; /**< property value */
} __attribute__((packed));

typedef struct mcm_property_info mcm_property_info;

/**
 *  mcm_property_info_linked_list
 */
struct mcm_property_info_linked_list {
    char                                 *propertyName;  /**< property name */
    char                                 *propertyValue; /**< property value */
    struct mcm_property_info_linked_list *next;          /**< linked-list */
} __attribute__((packed));

typedef struct mcm_property_info_linked_list mcm_property_info_linked_list;

/**
 *  Data structure for interface information
 */
struct mcm_interface_info {
    char interface_name[64]; /**< interface name */
    char device_name[16];    /**< device name */
    char device_type[16];    /**< device type */
} __attribute__((packed));

typedef struct mcm_interface_info mcm_interface_info;

/**
 *  Data structure for networking information
 */
struct mcm_network_info {
    bool     enabled;            /**< enabled */
    uint32_t wan_priority;       /**< wan priority */
    char     name[64];           /**< name */
    char     device_type[16];    /**< device type */
    char     device_name[16];    /**< device name */
    char     network_ifname[16]; /**< network ifname */
    char     network_type[16];   /**< network type */
    char     ipv4_method[16];    /**< ipv4 method */
    char     ipv6_method[16];    /**< ipv6 method */
    char     mac_address[64];    /**< mac address */
    char     err_reason[200];    /**< error reason */
} __attribute__((packed));

typedef struct mcm_network_info mcm_network_info;

/**
 *  Data structure for networking status
 */
struct mcm_network_status {
    bool     default_route;     /**< default route */
    char     ipv4_address[16];  /**< ipv4 address */
    char     ipv4_netmask[16];  /**< ipv4 netmask */
    char     ipv4_gateway[16];  /**< ipv4 gateway */
    char     ipv4_dns[2][16];   /**< ipv4 DNS */
    char     ipv6_address[64];  /**< ipv6 address */
    char     ipv6_netmask[64];  /**< ipv6 netmask */
    char     ipv6_gateway[64];  /**< ipv6 gateway */
    char     ipv6_dns[2][64];   /**< ipv6 DNS */
    uint32_t connection_status; /**< connection status (Indexed by ::McmConnectionState) */
} __attribute__((packed));

typedef struct mcm_network_status mcm_network_status;

/**
 *  Data structure for ethernet interface information
 */
struct mcm_ethernet_info {
    bool     link_detected; /**< link detected */
    uint32_t link_speed;    /**< link speed */
    char     duplex[16];    /**< duplex */
    bool     dhcp_server;   /**< DHCP Server status */
} __attribute__((packed));

typedef struct mcm_ethernet_info mcm_ethernet_info;

/**
 *  Data structure for wifi interface information
 */
struct mcm_wifi_info {
    char     bssid[32];          /**< BSSID */
    char     hwmode[16];         /**< hwmode */
    char     frequency[16];      /**< is 2.4g/5g */
    char     operation_mode[16]; /**<  operation mode */
    char     ssid[40];           /**< SSID */
    char     secure_mode[16];    /**< scure mode */
    char     signal[16];         /**< signal */
    uint32_t channel;            /**< channel */
    bool     broadcast;          /**< broadcast */
} __attribute__((packed));

typedef struct mcm_wifi_info mcm_wifi_info;

/**
 *  Data structure for modem interface information
 */
struct mcm_modem_info {
    char     cell_id[16];       /**< cell id */
    char     device_imei[16];   /**< device imei */
    char     network_rat[16];   /**< network rat */
    char     operator[32];      /**< operator */
    char     sim_iccid[32];     /**< sim iccid */
    char     sim_imsi[16];      /**< sim imsi */
    char     apn[32];           /**< apn */
    char     taclac[16];        /**< taclac */
    int32_t  pin_retries;       /**< pin retries */
    uint32_t signal_strength;   /**< signal strength */
    uint32_t sim_slot;          /**< sim slot */
    int32_t  modem_state;       /**< modem state (Indexed by ::McmModemState) */
    char     modem_name[64];    /**< modem name */
    char     modem_version[64]; /**< modem FWR version */
    int32_t  umts_rssi;         /**< 3G RSSI */
    int32_t  umts_ecio;         /**< 3G RSRP */
    int32_t  lte_rsrp;          /**< 4G RSRP */
    int32_t  lte_rssnr;         /**< 4G RSSNR */
    int32_t  nr_rsrp;           /**< 5G RSRP */
    int32_t  nr_snr;            /**< 5G SNR */
} __attribute__((packed));

typedef struct mcm_modem_info mcm_modem_info;

/**
 *  Data structure for wifi ap information
 */
struct mcm_wifi_ap {
    char     ssid[40];            /**< ssid */
    char     bssid[32];           /**< bssid */
    char     encryption_type[32]; /**< encryption type */
    uint32_t signal_strength;     /**< signal strength */
    char     frequency[16];       /**< freqency */
    int32_t  rssi;                /**< rssi */
} __attribute__((packed));

typedef struct mcm_wifi_ap mcm_wifi_ap;

/**
 *  Data structure for profiles information
 */
struct mcm_profiles {
    char profile_name[32]; /**< profile name */
} __attribute__((packed));

typedef struct mcm_profiles mcm_profiles;

/**
 *  Data structure for datausage information
 */
struct mcm_datausage_info {
    uint32_t tx; /**< TX Bytes */
    uint32_t rx; /**< RX Bytes */
} __attribute__((packed));

typedef struct mcm_datausage_info mcm_datausage_info;

/**
 *  Data structure for diag info information
 */
struct mcm_diag_info {
    uint32_t diag_state;   /**< diag bitmask state */
    char     message[512]; /**< diag message */
} __attribute__((packed));

typedef struct mcm_diag_info mcm_diag_info;

/**
 *  Data structure for diag info information
 */
struct mcm_upgrade_info {
    uint32_t state;        /**< upgrade bitmask state */
    char     message[512]; /**< upgrade message */
} __attribute__((packed));

typedef struct mcm_upgrade_info mcm_upgrade_info;

/**
 * @brief A help function for properies array free.
 *
 * @param arr_len (input) A number of properties array.
 *
 * @param info (input) A properies array being to free.
 *
 */
void mcm_free_property_info(int arr_len, mcm_property_info *info);

/**
 * @brief A help function for propertyInfo_sub appended to propertyInfo.
 *
 * @param error_num (input) A error number.
 *
 * @return error message.
 */
char *
mcm_get_error_message(int error_num);

#endif /* _MCM_BASE_INFO_H_ */
