#!/bin/bash -e
#
# Copyright (C) 2024 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#

set -e

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
source "${INSTL_SCRIPT_DIR}/lib/version.sh"
source "${INSTL_SCRIPT_DIR}/lib/driver.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

main() {
        log::info "Rename serial device with ttyMX"
        cat <<EOF >/etc/udev/rules.d/99-moxa-x86-serial-naming.rules
KERNEL=="ttyS0"   DRIVERS=="serial"  MODE="0660" SYMLINK+="ttyM0"
KERNEL=="ttyS1"   DRIVERS=="serial"  MODE="0660" SYMLINK+="ttyM1"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.1:1.0", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM2"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.1:1.1", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM3"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.1:1.2", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM4"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.1:1.3", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM5"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.2:1.0", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM6"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.2:1.1", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM7"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.2:1.2", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM8"
KERNEL=="ttyUSB[0-9]*", KERNELS=="1-4.2:1.3", SUBSYSTEM=="tty", ACTION=="add", SYMLINK+="ttyM9"
EOF
        {
                udevadm control --reload-rules
                udevadm trigger
        } >>"${INSTL_LOG_FILE}"

        # auto probe gpio-pca953x when boot up
        if ! util::is_exists "/usr/lib/modules-load.d/gpio-pca953x.conf"; then
                echo "gpio-pca953x" >/usr/lib/modules-load.d/gpio-pca953x.conf
        fi

        # auto probe hid_ft260 when boot up
        if ! util::is_exists "/usr/lib/modules-load.d/hid_ft260.conf"; then
                echo "hid_ft260" >/usr/lib/modules-load.d/hid_ft260.conf
        fi

        cat <<EOF >/etc/udev/rules.d/11-ft260-pca9535.rules
ACTION=="add", KERNEL=="0003:0403:6030.*", SUBSYSTEM=="hid", DRIVERS=="hid-generic", \
RUN+="/bin/bash -c 'echo \$kernel > /sys/bus/hid/drivers/hid-generic/unbind'", \
RUN+="/bin/bash -c 'echo \$kernel > /sys/bus/hid/drivers/ft260/bind'"
EOF
        cat <<EOF >/etc/udev/rules.d/12-ft260-systemd.rules
ACTION=="add", KERNEL=="0003:0403:6030.*", SUBSYSTEM=="hid", DRIVERS=="ft260", TAG+="systemd"
EOF
        log::info "Install hardware sensor monitor lm-sensors packages and configuration file"
        {
                case "${OS_PACKAGE_MANAGER}" in
                "apt") apt-get install -y "lm-sensors" ;;
                "yum") yum install -y "lm_sensors" ;;
                esac
        } >>"${INSTL_LOG_FILE}"

        cat <<EOF >/lib/modules-load.d/it87.conf
it87
EOF

        if ! util::is_exists "/etc/sensors.d/"; then
                mkdir -p "/etc/sensors.d/"
        fi

        if ! util::is_exists "/etc/sensors.d/it8786.conf"; then
                adc_lsb_coeff="0.9"
                V3_3_Rin="6.19"
                V3_3_Rf="10"
                V5_Rin="10"
                V5_Rf="10"

                cat <<EOF >/etc/sensors.d/it8786.conf
chip "it8786-*"
    ignore in0
    ignore in1

    label in2 "V3.3"
    compute in2 @*$adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)), @/($adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)))
    set in2_min 3.3*0.9
    set in2_max 3.3*1.1

    label in3 "V5.0"
    compute in3 @*$adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)), @/($adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)))
    set in3_min 5.0*0.9
    set in3_max 5.0*1.1

    ignore in4
    ignore in5
    ignore in6
    ignore in7
    # VBat
    ignore in8

    ignore fan1
    ignore fan2
    ignore fan3

    label temp1 "SYSTEM"
    label temp2 "SYSTEM2"

    ignore temp3
    ignore intrusion0
EOF

        fi

        log::info "Copy init file & enable init script"

        {
                pushd "${INSTL_TOP_DIR}/product.d/DA820E" >/dev/null

                cp da820e_init /sbin/
                cp da820e_init.service /lib/systemd/system/
                systemctl daemon-reload
                systemctl enable da820e_init

                popd
        } >>"${INSTL_LOG_FILE}" 2>&1

        if [[ "$LINUX_KERNEL_MAJOR_VER.$LINUX_KERNEL_MINOR_VER" == "5.15" ]]; then
                log::info "install Intel GPU Firmware package"
                {
                        wget -qO - https://repositories.intel.com/gpu/intel-graphics.key |
                                sudo gpg --yes --dearmor --output /usr/share/keyrings/intel-graphics.gpg
                        echo "deb [arch=amd64,i386 signed-by=/usr/share/keyrings/intel-graphics.gpg] https://repositories.intel.com/gpu/ubuntu jammy client" |
                                sudo tee /etc/apt/sources.list.d/intel-gpu-jammy.list
                        apt update
                        apt install intel-fw-gpu

                        rm /etc/apt/sources.list.d/intel-gpu-jammy.list
                        apt update

                } >>"${INSTL_LOG_FILE}" 2>&1
        fi
}

main
