(function (Random) {
  describe("engines.mt19937", function () {
    "use strict";

    it("is a function", function () {
      var mt = Random.engines.mt19937();

      expect(typeof mt).toBe("function");
    });

    describe("method: seed", function () {
      it("returns the same engine", function () {
        var mt = Random.engines.mt19937();

        var actual = mt.seed(0x1234);

        expect(actual).toBe(mt);
      });
    });

    describe("method: seedWithArray", function () {
      it("returns the same engine", function () {
        var mt = Random.engines.mt19937();

        var actual = mt.seedWithArray([0x1234, 0x2345]);

        expect(actual).toBe(mt);
      });
    });

    describe("method: getUseCount", function() {
      it("starts at zero", function() {
        var mt = Random.engines.mt19937();
        expect(mt.getUseCount()).toBe(0);

        mt.seedWithArray([0x1234, 0x2345]);
        expect(mt.getUseCount()).toBe(0);
      });

      it("increments with each generated random number", function() {
        var mt = Random.engines.mt19937();
        mt.seedWithArray([0x1234, 0x2345]);

        for (var i = 0; i < 1000; i++) {
          expect(mt.getUseCount()).toBe(i);
          mt();
          expect(mt.getUseCount()).toBe(i+1);
        }
      });

      it("increments with each discarded random number", function() {
        var mt = Random.engines.mt19937();
        mt.seedWithArray([0x1234, 0x2345]);

        mt.discard(1000);
        expect(mt.getUseCount()).toBe(1000);

        for (var i = 0; i < 1000; i++) {
          expect(mt.getUseCount()).toBe(1000 + i);
          mt.discard(1);
          expect(mt.getUseCount()).toBe(1001 + i);
        }

        mt.discard(25);
        expect(mt.getUseCount()).toBe(2025);
      });

      it("can be used to resume from a previous chain with the same seed", function() {
        var mt = Random.engines.mt19937();
        mt.seedWithArray([0x1234, 0x2345]);

        for (var i = 0; i < 1000; i++) {
          mt();
        }
        var mtUseCount = mt.getUseCount();
        expect(mtUseCount).toBe(1000);

        var newMt = Random.engines.mt19937();
        newMt.seedWithArray([0x1234, 0x2345]);

        newMt.discard(mtUseCount);

        expect(newMt()).toBe(mt());
      });
    });

    describe("method: autoSeed", function () {
      it("returns the same engine", function () {
        var mt = Random.engines.mt19937();

        var actual = mt.autoSeed();

        expect(actual).toBe(mt);
      });

      it("calls seedWithArray given the results of generateEntropyArray", function () {
        var mt = Random.engines.mt19937();
        var dummyArray = [1, 2, 3, 4];
        spyOn(Random, "generateEntropyArray").andReturn(dummyArray);
        spyOn(mt, "seedWithArray");

        mt.autoSeed();

        expect(mt.seedWithArray).toHaveBeenCalledWith(dummyArray);
      });
    });

    function times(count, callback) {
      var result = [];
      for (var i = 0; i < count; ++i) {
        result.push(callback(i));
      }
      return result;
    }

    function map(array, callback) {
      return times(array.length, function (i) {
        return callback(array[i]);
      });
    }

    function toInt32Array(array) {
      return map(array, function (x) {
        return x | 0;
      });
    }

    function test(seed, expectedValues, discard) {
      expectedValues = toInt32Array(expectedValues);
      it("passes tests for seed = " + seed, function () {
        var mt = Random.engines.mt19937();
        mt.seed(seed);
        if (discard) {
            mt.discard(discard);
        }

        var actual = times(expectedValues.length, mt);

        expect(actual).toEqual(expectedValues);
      });
    }

    function testArray(seed, expectedValues, discard) {
      expectedValues = toInt32Array(expectedValues);
      it("passes tests for seed = [" + seed.join(", ") + "]", function () {
        var mt = Random.engines.mt19937();
        mt.seedWithArray(seed);
        if (discard) {
            mt.discard(discard);
        }

        var actual = times(expectedValues.length, mt);

        expect(actual).toEqual(expectedValues);
      });
    }

    test(0x12345678, [0xc6979343, 0x962d2fa, 0xa73a24a4, 0xe118a180, 0xb5475abb, 0x64613c7c, 0x6f32f4db, 0xf27bf199, 0x464dd8dc, 0x95c1fed6, 0xb6df5ff8, 0x82b905ca, 0xc549443, 0xe7187f67, 0xea04d0f, 0x8a77ffb1]);
    test(0x522f1a98, [0x656de610, 0xf9f66cbb, 0xca483ab3, 0x5cc89323, 0xc0b3add5, 0xb5ef850d, 0xc1df576f, 0x99c6e4e8, 0xc0fa87da, 0x49332e6c, 0x758ea02d, 0x37fb01de, 0x9dfff197, 0xfb8ca9b5, 0x16d070f5, 0x7ae79d05]);
    test(0x494e9ae8, [0x32d80f38, 0x3213e352, 0x2e4dbaac, 0x30525ddf, 0x9ab54be3, 0xbb2a60e9, 0xcb66ae4a, 0x65b0736, 0x3ba48edb, 0xb611a917, 0xfebe7ed2, 0xd8e93a8c, 0xb4ee1746, 0x9bb2dda3, 0xee409f1e, 0x63e7e93d]);
    test(0xef97b6ff, [0x236e5ff3, 0xdcb55324, 0x9c3aa2f0, 0x1bb8089, 0x953b982f, 0x56f760aa, 0x972af66, 0xad22b89a, 0xd29e83f1, 0x14d030b7, 0x9c9fd9bd, 0xb8965602, 0x13a2bf88, 0x47750566, 0x878adc28, 0x69eb7d3d]);
    test(0xbe6df956, [0xd1d1065f, 0xc1ae23b3, 0x98876b5b, 0x4ad68baa, 0x47441e33, 0x97de9b2, 0xe4047131, 0x986960ca, 0xad16dc89, 0x10a438a2, 0x1315dd59, 0x7d4e5240, 0xd68da64, 0xdf421fe8, 0x66412745, 0x7c6536b8]);
    test(0x8b95a1a0, [0x8652f888, 0xedc5dba6, 0x4c507856, 0x46bacc1, 0xa47f4fb2, 0x46f6c605, 0x940be0f5, 0x587aafae, 0x7de18be1, 0x7d475dc3, 0x7e80b6ea, 0x7f7587fd, 0x5556898d, 0x9684cbc2, 0xec0246d9, 0xfc7fb408]);
    test(0x1d28981d, [0x9279184f, 0x35a9219a, 0x2c192470, 0xf2fa158e, 0x23063711, 0x42a7634d, 0x8de6c065, 0x6af9beea, 0x899a9e52, 0x90f34ef, 0x4607457d, 0x712c1829, 0x16ca8d28, 0x952cd2a4, 0xa25c0bcf, 0x3d2ec4e4]);
    test(0x1, [0x6ac1f425, 0xff4780eb, 0xb8672f8c, 0xeebc1448, 0x77eff, 0x20ccc389, 0x4d65aacb, 0xffc11e85, 0x2591cb4f, 0x3c7053c0, 0x17a38090, 0x65865081, 0x2faebfcc, 0x634e1e47, 0x5876aaed, 0xab7479fc, 0x65928d86, 0xef7f7d19, 0x89efe4b2, 0xd8a7d514, 0x6b5054fe, 0x5032b165, 0xaf6a8b92, 0x8648c9d4, 0x3456fb8b, 0x718620fc, 0xe0cc4dea, 0x3ac5929c, 0x702df9d, 0x88cf598e, 0xaba3c232, 0xe9f96a44, 0x6ad47cd7, 0x750b5fd7, 0x8f064be9, 0x6e4242f1, 0x23f065f7, 0xf06aadde, 0x32b6c760, 0xc7448456, 0xccfd988d, 0xb749d7e9, 0xf7dffd89, 0xcd818407, 0x503c913f, 0x17c1cb3d, 0xb13c0e16, 0x84a5a539, 0xe05b0a01, 0xdd71f780, 0xe504f13c, 0xd442f8d1, 0x15c57508, 0xd460e2d8, 0x9ff7e8d, 0x45e69a73, 0x2b7a01af, 0xf2a8fea, 0xe0cdf279, 0xaba7b9c8, 0x192d421e, 0x97d32447, 0x6bcdb583, 0xabf185c6, 0xf5383f95, 0x696aee31, 0x887d8539, 0x3292b203, 0xb11edbc4, 0x4a252b18, 0x50c5a1f1, 0x2461fc2b, 0xafbe864c, 0xc8874c1a, 0xd5aa0734, 0x699c2550, 0x4ae8a6d, 0x8bf7373, 0xc0097529, 0x9fc06dd2, 0xfd26000f, 0xa91f6c40, 0xbf87c8c4, 0x4c6a3019, 0x47cb2d6f, 0x7235dee2, 0xca0e35d7, 0x38dd2787, 0x1a6d051a, 0x12c7fe99, 0x72a92668, 0x78200416, 0xe899b709, 0x189ebec3, 0x4b2a4be7, 0xe743447e, 0x49aba517, 0x1e96ed7d, 0x21498d64, 0x8659409b, 0x4f53ba5, 0x15685139, 0xadc82a53, 0xeab76ca6, 0x362d4288, 0xe9132520, 0x43fadda2, 0x4c86af0a, 0x7dd7bd17, 0x959a868f, 0xda92aef, 0x90df9c57, 0x92f95f19, 0x9d2b0fc7, 0x259000f3, 0xf4df855c, 0x96dcba4a, 0x42cf84be, 0xb3235d2e, 0x3b23d3a0, 0x1a3296d8, 0x88901497, 0x69ff92b7, 0xf32f2541, 0xb1c43571, 0x7e39274d, 0x6a07a703, 0x8a64cb80, 0xcc9bff8, 0xc3f6d4fd, 0x893081cd, 0xb9bc706, 0xa9ee7234, 0x23d4d1d5, 0x83cfc5c6, 0xcadaf602, 0xf1d0f64c, 0x7a10fdb, 0x96287895, 0xe21482cb, 0xe7455907, 0x8a71184d, 0x23318ac8, 0x72aed44b, 0x23a79d4c, 0xe463042b, 0xceb13214, 0x60a95e1e, 0x65ce2624, 0x89d63367, 0x2a54a707, 0xa6fd0f2d, 0xed7133c4, 0x5c7b9a39, 0x59072ef0, 0x922d9dfc, 0xc03538d2, 0xa3494060, 0xb9db010d, 0x20562c0a, 0xe2205917, 0xb0b13f7c, 0x9fa8fb51, 0xa5d2e787, 0xc03dc379, 0x5a9bc098, 0x595166ca, 0xc3633ddc, 0x4519fe94, 0x5b45a9a0, 0xe558cc8c, 0xc0b6bcc1, 0x6d97625e, 0xe19f9e3c, 0xf6ffc1e9, 0x2fcc098, 0xa9d74d52, 0x7f841373, 0x9f277361, 0x12e40882, 0x1d5ffdec, 0xc975a6dc, 0xf311ba67, 0x1066b762, 0x732d710a, 0x5af59eb6, 0x94115760, 0xf11c39e9, 0x687ba752, 0x613ac9d6, 0x3cadccc6, 0xc34ebac2, 0xe743e147, 0xc5873fe7, 0x92dca8b0, 0x4d25f636, 0xbc1c0f, 0xc5d23b85, 0x9dfd3591, 0x272668aa, 0x539f0014, 0x94215a76, 0x86ed47b0, 0x24e5916, 0xe2cd19e5, 0xb583cf8d, 0x5b7607f1, 0x787bea72, 0xe895c261, 0xc3bc56b5, 0x9f948754, 0x77a70c0a, 0x40cdc60, 0x44dd40b7, 0xedef993d, 0xd4e9ce38, 0xb0de9ed9, 0x8d23a195, 0xff508ce7, 0x11f0dc60, 0x2c1e81f9, 0x78f41fd3, 0x231b5419, 0xbe25d5f1, 0xeebe938e, 0x3124088d, 0xb262acd4, 0x76dd1e74, 0x10e5632b, 0x3af90e86, 0xc16606cd, 0x821c82b8, 0xc0fe07bb, 0x3564bb8f, 0xec4b5618, 0xc9f33fb, 0xb6267c89, 0x84d421c2, 0x1fd038c7, 0x2c1a6f35, 0x516dd45, 0x6574c9a4, 0x6b5c365, 0x1b7c4af8, 0x73f1815, 0x8282ed28, 0x3f07b04d, 0x1a2588db, 0xdc2acab1, 0x49e96971, 0x89f0d52f, 0x3b5485cd, 0x8d85bda8, 0xf7b915ce, 0xd78f562d, 0x473fe757, 0x1fc9d290, 0x3bceaf9c, 0x477894ea, 0x17703e2d, 0x95f451c3, 0x91efe374, 0xf8376d42, 0x6afd3bce, 0x8f9fad2e, 0x5e2af980, 0x4c6119d, 0xd020703f, 0xccf6434b, 0x4a2db623, 0x3ba433b5, 0xb7a6f9dd, 0xce9e7221, 0x9cea2a82, 0x634ad5d4, 0x6d49f653, 0xdd111430, 0xc07abf36, 0xbf435d20, 0x6d8520fd, 0x8e65c29c, 0x6daa7837, 0x22eebad2, 0x5ca5c51f, 0xf56c41c, 0x274130de, 0x1f105d4a, 0xefebb488, 0xb67c06d, 0xea99d663, 0x1b84bc20, 0xc82e6e08, 0x39c81654, 0x9dc7bbcd, 0xb6867232, 0x937424f, 0x8f499ca9, 0xa781c340, 0x336de6c, 0x213894d3, 0x126ce818, 0x4b5b2471, 0xf79f6bea, 0x5c5e5b14, 0x916f082c, 0x464f250f, 0x340b069e, 0x12efa1db, 0x40986b8e, 0x26f275f3, 0xbe6b5f13, 0x295fd4fb, 0x3207aa9a, 0xf04f68eb, 0x94d3f056, 0x5f02a187, 0xf8533ae3, 0xccdfc35, 0xd8c9c5af, 0xbdaf436e, 0x3d66a9bc, 0x4c764ef3, 0x7e67b122, 0x3450d064, 0x9eb56ae4, 0xfb546720, 0xd4381793, 0xe49d09c3, 0x28237b18, 0xc264aed3, 0x4c168e5, 0xa6afd65e, 0x11ecf8a6, 0xa28b2af, 0x7c811ce7, 0xd05a3385, 0x9b38684f, 0xc337fe3f, 0x91a03f71, 0x748d6457, 0x513ea9a0, 0x8825c72a, 0xfd15f2ca, 0x1bfd72c2, 0x946a2ed8, 0x256266e2, 0x6150ee9e, 0x823ffa91, 0x8d0af144, 0x3734db40, 0xbece3cbc, 0xea9c0df4, 0xab52d8ce, 0x764fa091, 0x43d1c4a7, 0x21dc44a3, 0x10fb51d1, 0xc3a5569c, 0x5ebdd696, 0x3675b726, 0xa1352aa9, 0x130299ca, 0x35cdf6cd, 0x186d6546, 0xc0b49699, 0x29864ce6, 0x110888f0, 0x5b413430, 0x42a402b2, 0x186617be, 0xce04652c, 0x24ace269, 0x3184e8b6, 0xee56dd00, 0xa3b3b556, 0x939b05f7, 0x8650cb10, 0xd6f2fd13, 0xecc03709, 0x9f901270, 0x43676adc, 0x53139433, 0x10e2d36f, 0xba5ef88a, 0xbc2d487b, 0x85d21144, 0xc5ad7597, 0xbc9fad0e, 0xe8669ebf, 0x2a580e15, 0xee95b8ae, 0xafe30d7c, 0x3925483, 0x6d4379b8, 0x3bff2758, 0xba82bcd0, 0x9de52fae, 0xc19f3236, 0xf2f2bbcf, 0x65c9da47, 0xf33ebdfc, 0xecd9ea0e, 0x8e80d2cd, 0x3419408f, 0xea652d99, 0x20c7735, 0xa43daed4, 0xed2551f9, 0x63d78bba, 0x4b651bd5, 0x7c69e272, 0x2abd56ee, 0x9ab4179d, 0x62b862c, 0x8caf2c25, 0x73b66096, 0xed1a39db, 0xceef49b6, 0xeb321d4c, 0x5e4df20c, 0x65169178, 0x9bf4fd3b, 0xf6985f9a, 0x8ebc7cc, 0x2c885bc7, 0x5ac592d7, 0x205721a7, 0x1419e27d, 0x22948c2b, 0xb17495cc, 0x81731372, 0x3412326, 0x582a65b, 0x75a4c270, 0xf2ae2cfd, 0xf618e345, 0xd3bdd6e2, 0x558d29a1, 0x3d848ab, 0x78da759a, 0x2d1b32b8, 0x1afaebc5, 0x55021e49, 0x80c99534, 0x21890259, 0xe2bc921b, 0xcf3ac82b, 0x88ccf401, 0x5840a952, 0x480edc7f, 0xf0aae25e, 0x5ac60fe0, 0x94fee19a, 0xe5729f54, 0xe0fb221e, 0x3dd23b67, 0xd8408440, 0x61d4496, 0xe7c7ca62, 0xf739df34, 0x75bab683, 0x6e126046, 0x8bdd628c, 0x58a65d27, 0xcc7148e6, 0x93bab453, 0x4924deef, 0x206404b0, 0x7d81413d, 0xf33609c6, 0x995f4b0d, 0x5048cca4, 0x3f9fd17, 0xf3ecbd96, 0x97ee65c2, 0x37e89e7a, 0x6f0569dc, 0x3faad0f9, 0xceaf2dea, 0xdd221d5f, 0x50b3e235, 0x3c3a8b89, 0xe4945aa9, 0xd0aa65b9, 0x93ee7358, 0x8b3cd96d, 0x2f1b4af6, 0x30f66edd, 0xc9b5baf4, 0x96be36cc, 0x9cae1352, 0xc52b16a, 0xdccff8b, 0x4104069, 0x6b91d024, 0xcb9dc70, 0xadd77490, 0x6640149e, 0xeb297c73, 0x93affa6a, 0x1a58d4, 0xde159fb9, 0xfa0ce323, 0xc941eda9, 0x60679151, 0x41343302, 0xf949e0d5, 0x150ea85d, 0x9aceaca4, 0x2862b26d, 0xd42f3d2e, 0x35a15350, 0x93204b1b, 0x6ad99948, 0xa0c99a1c, 0x59023682, 0x491b86e3, 0xb32b998d, 0x963ab5a9, 0xb28ffb79, 0xc0016d24, 0x13211ae4, 0xdbba74a8, 0x9561412, 0xc14d10e1, 0xca161ba6, 0xb2b3e13f, 0xe68909f3, 0xdd4e862e, 0x1319d68, 0x529b38d9, 0x43a2d3e7, 0xabb8d070, 0x77a609ff, 0x736c7965, 0x5c1cea02, 0x61d17c62, 0x3d7ee71c, 0x692aeece, 0x5ee62cb, 0x66c75dba, 0x881b66aa, 0x5140131a, 0x215470e8, 0x9f361b95, 0x8fbbc63a, 0x6e24af7f, 0x2af6f590, 0xf94b17dc, 0x6bcb8e25, 0xad845be9, 0xaf47b4ed, 0x32d579e8, 0x461e54ea, 0x6d3c470b, 0x28f5796, 0x57e58a06, 0x38492860, 0xcc320e8e, 0x82e4864, 0xe1479157, 0xef53c8cb, 0xe7622fa3, 0xc8372ebf, 0xa9a8017b, 0x84810a20, 0x452c5e6e, 0x436e75b, 0x409b1ab1, 0x4c7e2be6, 0xdada977a, 0x8a6c9880, 0x87184e96, 0xdc215461, 0xcd5a6dc6, 0x2f63a6a4, 0x928e9b92, 0x15634b1b, 0xbbaf3a9c, 0xb556de5, 0x84ddf234, 0x719d6666, 0xc558a5f2, 0x4f212de, 0x91a0ad79, 0x11290332, 0x7738c32e, 0xcfd04207, 0x57ba75d5, 0xa2e3f110, 0x11762ae7, 0xc5d9869d, 0x60bfa38b, 0xb4a795bd, 0x14625ffb, 0x3cfb63fa, 0xfb99e70e, 0xb5ef968, 0x2e7e2e1e, 0x47fdd488, 0xcfd5f8a8, 0xdd7683b0, 0xdffd7c9e, 0x12b83597, 0xb03bd9c2, 0xa90b82dc, 0x91ca62c0, 0x2ff80ae7, 0x29356c8c, 0x85b49291, 0x778572fa, 0xe764a1a1, 0x585d320a, 0x79a39c1c, 0x399c37e9, 0xd8d673cb, 0x97aedb85, 0xd6ba3f51, 0x4ff0ea85, 0x23a7db79, 0xea93002a, 0x8a516656, 0xe8dddfb4, 0x7792e739, 0x41d28138, 0x37fb9a4e, 0x1c635f57, 0x70da1ffa, 0x31660175, 0x7af382d1, 0x7fe4bf8a, 0x55521ec8, 0xba849730, 0x24819f5, 0x354c3b13, 0x8166158c, 0x3f7f2099, 0xac53614d, 0xda072b10, 0xe329504, 0x6a750fd8, 0x28572376, 0x9ddf129b, 0x9ae80bb2, 0x3bd18b44, 0x51b8a683, 0x1a1a86ba, 0xe3f8ee0c, 0x840f3482, 0xca55eb4c, 0x7a25e960, 0x37ae0760, 0x27157d3d, 0x3340e58f, 0x9f2eb164, 0xae3151fa, 0x8b443f4a, 0x27fdd38c, 0xa7758b92, 0x82a7049e, 0x2500efbb, 0xfcf1ce85, 0xc0642090, 0x250a875f, 0x38d83660, 0xc5d63bc, 0x84f43dab, 0xe6290eec, 0xc909291f, 0xb71fd6eb, 0x5b77263, 0x4f71a52c, 0x53096b18, 0x6497bbbe, 0xdf77d74f, 0x14ae3e15, 0xd83ee393, 0x5a249f4d, 0x89d73e21, 0x426804, 0xddda0a27, 0x9fe7c87c, 0xf3267c43, 0x3344fca2, 0xd38f68a2, 0xa007502a, 0xdaa74f4e, 0x524aaad2, 0x19473f66, 0xc1d0c26, 0xa6bbe17a, 0xb92e6eb0, 0xb419b07a, 0xc25e56d1, 0x9c38bdfe, 0x68038f05, 0xccb395fa, 0xb44f4ce6, 0x8d9a8db, 0x95da19f5, 0xc52e5d88, 0xf8356128, 0xbb5290c5, 0x4dcafa05, 0x427b9807, 0x505245ce, 0x41cf4b25, 0x39f96369, 0xa1dea14e, 0xac1e1e21, 0x58656a1b, 0xc76772b3, 0xcbed3c4d, 0x833dcc1e, 0x7236a3ab, 0x9d8200ae, 0xc8624409, 0xbb07343, 0xfd8f8f03, 0x723393c3, 0x4cdd1336, 0xf7cfad61, 0x249c07bf, 0x5c0c24e3, 0xe6bc2650, 0x8dd50d1c, 0x8aa3a2a5, 0x694e85c0, 0xf98895a5, 0xf74ed3dd, 0xa2f88192, 0x33fb994e, 0xfe711567, 0xfe29e1c6, 0x8bcb4bdc, 0xa8c15434, 0x86c3d98a, 0xcf1fafa0, 0x22ab672e, 0x7549603f, 0x5b0f7e7b, 0xe3269edd, 0x6b64292, 0x88afeba4, 0x290fa89d, 0x93c895f8, 0xbee21427, 0xeae12b1c, 0x7c84636, 0x506be81d, 0x5dd5c4b7, 0xb9e43567, 0xdcc2b95a, 0xce23ef9b, 0xb15353bc, 0x7ae5676a, 0xb0e1958c, 0x7cd1ac91, 0x304a8047, 0xd766a65b, 0x7120a387, 0x8e370d7c, 0x94e241d3, 0x4516f0b8, 0xfd605e33, 0x7c9b2ba0, 0x343332cc, 0x469dd63e, 0x3f6b6c66, 0x39b330b3, 0x431dc660, 0xa591aa1e, 0xc00b4c96, 0xebcf6905, 0x74fc55db, 0x3352505f, 0xe92ed69, 0xe13e0d96, 0x822e1ecc, 0x773f2c74, 0x3643056d, 0xb9984c16, 0xcc7153e6, 0x4d88f992, 0x4c1de8dd, 0x7d2f951d, 0x7113012, 0x34240034, 0x97eb307a, 0x16af8e1f, 0xd805ed31, 0x2a6aa2e2, 0x618a45d3, 0x80c46179, 0xbff6b6dd, 0xc99ee8ad, 0x82da2aeb, 0xe75273b3, 0x8a7bd156, 0x3832bb0f, 0xf59d7cc7, 0xbcf626b0, 0xcdd0619c, 0x41ff5ed3, 0x846531b, 0x906ffbfd, 0xb59a672b, 0x21e6e893, 0x770a564f, 0x5f412346, 0xf2929158, 0x153603fa, 0x38afd0d6, 0x8177460b, 0x445ed4e5, 0xb1c4a5c1, 0x14db7a5d, 0x74fc810f, 0x6db9f6b3, 0x5c7c7fe2, 0x1be8a756, 0xd0e5de2d, 0xa23fd973, 0xbba4dbf, 0xcd8effaa, 0x6348ae96, 0xb261846d, 0x7561c667, 0xc4266dd2, 0xbc58f0a8, 0x57ab12ac, 0xe8ddf790, 0xd889b911, 0x47040b6b, 0x6dc3ca58, 0xd15944e6, 0xd2f24f80, 0xa80450f0, 0xa0620d61, 0x2e34452, 0x24b75f77, 0xbc4023c7, 0x141129f4, 0x9a7ea00e, 0x4b172a4, 0xae15bc1a, 0x1114e3a0, 0x957e6381, 0x7565bfb0, 0x92d24558, 0x1d03f9f5, 0x5c6ba416, 0x71ccf22, 0xf56d5b5e, 0xc13e9a15, 0xfca848c4, 0x6514ebc3, 0x631b4eb0, 0xbf375c76, 0xd60fe492, 0x73d0cd61, 0xfb6dae8b, 0x7338e28b, 0x95d9b179, 0x7a62f5b4, 0xabc2334c, 0x79585244, 0x1614e8ad, 0xcd9c1cc0, 0xfdcfb459, 0x67033228, 0xc78d5307, 0xe7998307, 0xfe8eb0f6, 0x97cd521, 0xa758363a, 0xc61ca107, 0x994e0a2, 0x202a0884, 0xb8163d85, 0x9e56e7bd, 0xb0286076, 0x2a73b67, 0x9bdb2cb2, 0x89e37a4c, 0x1b30a9ad, 0xc5c8fe, 0x47daf560, 0xf3816f95, 0xc6b60072, 0xe7c86d7d, 0xe3e1f05c, 0xcbc47d47, 0x82b6a2dc, 0xea4f6aef, 0xb738a299, 0x25434de9, 0x13e950f7, 0x2860ff61, 0x9d849401, 0x3008a10d, 0xd67a5074, 0x9f5be426, 0xb22d68d5, 0xe7e3219b, 0x8e185ea6, 0xfd6db3d4, 0xdc76eca1, 0xb60c1f01, 0xbecd63fe, 0xbb57459f, 0x43b980f5, 0xe8c77071, 0xfff0de72, 0x669fa926, 0x92e0181c, 0x3ff636cb, 0xa3995e72, 0x2c65eb63, 0x7d5cf41f, 0x1e94bcb3, 0xe3ae1042, 0xd0073f43, 0xdfdfe7a0, 0x25942f4a, 0xd908c3a0, 0x43a8ffe1, 0xa5d13b05, 0xd1afd400, 0x95e87ba5, 0x4f82a56e, 0x4815cf48, 0xfb7fb5ab, 0xfddd096c, 0x44426f18, 0xccb69890, 0x889d817d, 0x806b3b60, 0x5080e8e0, 0xc5b67529, 0xe9286898, 0x1fd0d115, 0x5dd6a7f9, 0xebc7d39b, 0x6effe82e, 0xb7a6923, 0x83259d3e, 0x3a85d844, 0xf05add3e, 0xd7ad2f7d, 0x7ec4620, 0x4e39f0f9, 0xb7855c2d, 0x7bfc7f58, 0xe419d166, 0xa669e973, 0x6fc4baa, 0x24d97039, 0x85a52699, 0x4e48cb20, 0x53741168, 0xe30d3821, 0xdc077df6, 0xe3081286, 0x8efaf0c1, 0x83e53d33, 0xb0b2c61f, 0x445fd987, 0x73ee34fb, 0xe15e308d, 0xa0d8dc7f, 0x7f7bd583, 0x4a43c99c, 0x555c468f, 0x264ab1d, 0xe50ad642, 0x93a646ef, 0x3459003d, 0x4fbaced3, 0xd1d19c60, 0x846ba63a, 0xd6be13d2, 0xea9992e8, 0x85d5f5fa, 0x6d2d73a2, 0x2974c31e, 0x3f5558b9, 0xf2f02860, 0x5f0d1bb1, 0xf380dfc, 0xee8e732e, 0xe2f1a153, 0xefd69b29, 0x93083914, 0xd82601f8, 0x60fc1c28, 0xeb92a778, 0xa4125dfe, 0x3a57ac6e, 0x20b7f5b2]);

    testArray([0xd930680f, 0x1334b65d, 0x19aa455c, 0xfae4429], [0xe8336da6, 0xdfea0fee, 0x375775b8, 0xa513cc7, 0x5b700dce, 0xe37faac7, 0x4b364eca, 0x7b7ad6e1, 0x95bf24b4, 0x1bcfeced, 0x35a97d0, 0xd2a31582, 0x7ec27cf, 0x7a9175e5, 0x86d515f3, 0x430ea031]);
    testArray([0x1234, 0x2345, 0x3456, 0x4567], [0x7a422abd, 0xd48bc506, 0x53454c8b, 0x1126affe, 0x5e471e42, 0xd9621c51, 0x1a3525c, 0xa29c2837, 0x363c307d, 0xdadc88a6, 0x825f582e, 0xa9d0c112, 0x4ccc4b24, 0x915b4faa, 0xab952efe, 0xaf798601]);
    testArray([0x2cfc5537, 0x1b17526d, 0x69f97a02, 0x7b2f9891], [0x570ae143, 0xf7d00324, 0xfa6fcf8b, 0xe1e08c63, 0xa5a5c4cd, 0xaabebadf, 0x7db637d9, 0x18189b01, 0x6024d03b, 0xd4c39ae6, 0x42797b3e, 0xc3a55da4, 0x1e56d432, 0x4851a7f3, 0x6a4fa9cd, 0x7be3b0a2]);
    testArray([0xa9990a6f, 0xdaaf2865, 0x36ccb58, 0xb667cbfe], [0x85640639, 0xbd705238, 0x7bb3a007, 0x4f3e1b92, 0xf553dad5, 0x325dd4c1, 0xe4eabfe5, 0x315e51f2, 0xe211075f, 0xcdfe712f, 0xbe24fa07, 0x5d2f6ee, 0x6a249a99, 0x5e9683f3, 0xfc2d9d4c, 0x24ef6118]);
    testArray([0xca6e67d8, 0xb9286516, 0x8828562, 0xa2defcf6, 0x82ba5f3f, 0x8f6c0a5f, 0x56b90246, 0xe9d0cbfa], [0x7a7b1750, 0x30f96caf, 0x4f6e5056, 0x3db7508a, 0x825a5289, 0xa6ae4d28, 0x8a3802ff, 0x945d7aa3, 0x7f3f8c06, 0x48eb0e69, 0xbefe6eb2, 0x75ec576f, 0x1bbacdaa, 0x6efe5acc, 0x98e8a667, 0xf16633ff]);
    testArray([0xda40418c, 0x4dd1b680, 0x6da0160d, 0xea2f8c00, 0xfab513e5, 0xd86980fe, 0x99162236, 0xccef50d1], [0x860a91ea, 0x3c842673, 0xcb91498d, 0x4a68a854, 0x666f29a3, 0x253dbae6, 0x16796646, 0x6a8fddc5, 0xead31993, 0xaceb1665, 0x200ea484, 0xf3bf21c7, 0xe2543cf4, 0xda188548, 0x9b71dcaf, 0xcee06122]);
    testArray([0xcd1f7b1, 0x6b7a2d23], [0xc41d290, 0x2a01cf19, 0x3280f35f, 0xddfbed6a, 0xedee289b, 0x222b2367, 0xf2dae262, 0x3370df25, 0x3e893999, 0xbcb5d93b, 0x1eee9a97, 0x3527335e, 0x2ffa88d1, 0xbde2e1f3, 0xaf945f67, 0x46f1739a]);
    testArray([0xd8ccf3fd, 0xed85db28], [0xea402254, 0x4be695cf, 0x159a6f84, 0x1d0e68ca, 0x4c787dba, 0x69717c12, 0xd18f1aaa, 0xc6f8ee84, 0x9fde16af, 0xb120f696, 0x26a47477, 0xb81d4f0a, 0x5616de6c, 0x1101efdb, 0xf0dcfbe4, 0xff350f41]);

    test(0x1, [0x75c24bc2, 0x4330b935, 0x8080568a, 0x8fd8c12, 0x863a41de, 0x184fe782, 0x6c3217be, 0xb00e264, 0x47ee6a1f, 0xe2a71a3b, 0xa42cd40d, 0xac9d4f88, 0xa2694266, 0xa2c0b37b, 0xda5a54d7, 0x7ebf22bb], 1000000);
    test(0x12345678, [0x1a843c4b, 0x77feca9d, 0x78f045ab, 0x63fb95d0, 0x4a26807d, 0x2de39aed, 0xc549be4b, 0x71adac91, 0xbded7218, 0xdaade893, 0x1d69c9c1, 0xa2c4f153, 0x88841d1a, 0x50d67900, 0x43e90fa6, 0xa0c706c], 1000000);
    test(0xa859ea05, [0x2d764ce1, 0xb79c423e, 0x7e4d9909, 0x7b7a2098, 0xdc8b61bb, 0xf5357aeb, 0xabcfd521, 0x9ba49331, 0x1ed7a6b5, 0xf9d8110b, 0xbd4494de, 0xe58cc2e5, 0x3e18ce79, 0x88074f39, 0xd5f13134, 0xd8583007], 1000000);
    test(0xe906a042, [0x58f427b7, 0x24ce2e63, 0x150fecd, 0x670a0316, 0x10d91dcb, 0xdc961a07, 0x96833126, 0x57a3221, 0x17fa7814, 0x38617d98, 0x2bfbae67, 0x4973b6ca, 0xac2cf2e2, 0x26c31922, 0xd8e37427, 0x854f5b24], 1000000);
    test(0x1c8c67df, [0x853d9eb0, 0xa8f3e136, 0xa381927e, 0x411164b6, 0xa32a5ac8, 0x58214288, 0xacd4d3eb, 0xa2a92c81, 0xb425a8f8, 0xfb21e2f7, 0xd4427e34, 0x2051e3d5, 0x6a44d73f, 0x4c942513, 0x755bbe4f, 0xa2ee1587], 1000000);
    test(0x397e437, [0x475e0fd0, 0xb7352b5a, 0xf1f3fe70, 0x955304b0, 0x459c0b21, 0x41f1e13c, 0x97b92317, 0xdad128b3, 0xeae85768, 0x3f3e3d08, 0x3fb2df1c, 0xb85da862, 0x3e8799a9, 0xc2b00386, 0xae0573bb, 0x36c641a9], 1000000);

    testArray([0x10cd7cbe, 0xa90b3040], [0x97bf6355, 0xab1bf842, 0x82ee8b4c, 0x8fe45605, 0x624efd53, 0x361b95d8, 0x861e0c3f, 0x80cf514d, 0x61174569, 0x2eca5151, 0x5a53d78f, 0x54f48e5f, 0x5e4bac3, 0x1958bc37, 0xfc0d6736, 0xaa47d9dc], 1000000);
    testArray([0xd036b7f3, 0x1775082e], [0x35f5c874, 0xf8f1b9eb, 0x13fb4629, 0xdd9ea347, 0x2c5cc50, 0x7a607b2a, 0x12b44309, 0x4f80c0c0, 0x346ebd92, 0x9bc61c1b, 0x48b8b08c, 0x3663cb5, 0x6aef614a, 0x11ef1adf, 0x26d423ca, 0x7d0e77c5], 1000000);
    testArray([0x11aded04, 0x8f5d4242, 0x53b4dd38, 0xdf64d789], [0x599c4343, 0xae2a4d3e, 0x71abb9b1, 0xee1a2ac4, 0x8257fed1, 0x87b2544e, 0xd7576e47, 0x2f5a55fd, 0xa7c40d53, 0x4a01078d, 0x3ee20d96, 0xbcc673fe, 0xffc2fa4e, 0x9eda95ca, 0xe06a868c, 0xf382542e], 1000000);
    testArray([0x57ef9b89, 0x381c2de5, 0xdd56ff9e, 0xe94f3279], [0xaa3a47e6, 0x9ad7e49f, 0xddacdcd2, 0x7af6988b, 0xa9894b8f, 0x38926151, 0xdbad2e5f, 0x7d598e3a, 0xc230300e, 0xfafccbad, 0x7f39347c, 0x79deee2e, 0xf3f09de4, 0x21fa695b, 0x457ace3c, 0x3a2613da], 1000000);
    testArray([0xc4f5638, 0x9c81aef4, 0x8cd3091d, 0x12d5160a], [0x566178b9, 0x96c41b7, 0xd4ab9f2b, 0x6420a26, 0x214ebdfb, 0xa95f6418, 0x9544fb2d, 0xf73413e8, 0xa0095c75, 0x1047500f, 0x59867add, 0xa478b2e1, 0x7ea93c01, 0xb2fa9ec3, 0x559095a4, 0x984447b4], 1000000);
    testArray([0xa6db0297, 0x16afeb6, 0x340be4b4, 0x930b3922], [0xb0885d11, 0xb445e8e9, 0xc725c250, 0xe96ea2c1, 0x1aa513a5, 0x3b101e51, 0xd3d4b908, 0xfaee614e, 0x2541ee94, 0x239e8c7e, 0xae7e9de7, 0xfa9fdbdc, 0xff4d8b58, 0x9b06f428, 0xb9ca9090, 0xc9e805c0], 1000000);

    test(0x1, [0xff4780eb, 0xb8672f8c, 0xeebc1448, 0x77eff, 0x20ccc389, 0x4d65aacb, 0xffc11e85, 0x2591cb4f, 0x3c7053c0, 0x17a38090, 0x65865081, 0x2faebfcc, 0x634e1e47, 0x5876aaed, 0xab7479fc, 0x65928d86, 0xef7f7d19, 0x89efe4b2, 0xd8a7d514, 0x6b5054fe, 0x5032b165, 0xaf6a8b92, 0x8648c9d4, 0x3456fb8b, 0x718620fc, 0xe0cc4dea, 0x3ac5929c, 0x702df9d, 0x88cf598e, 0xaba3c232, 0xe9f96a44, 0x6ad47cd7, 0x750b5fd7, 0x8f064be9, 0x6e4242f1, 0x23f065f7, 0xf06aadde, 0x32b6c760, 0xc7448456, 0xccfd988d, 0xb749d7e9, 0xf7dffd89, 0xcd818407, 0x503c913f, 0x17c1cb3d, 0xb13c0e16, 0x84a5a539, 0xe05b0a01, 0xdd71f780, 0xe504f13c, 0xd442f8d1, 0x15c57508, 0xd460e2d8, 0x9ff7e8d, 0x45e69a73, 0x2b7a01af, 0xf2a8fea, 0xe0cdf279, 0xaba7b9c8, 0x192d421e, 0x97d32447, 0x6bcdb583, 0xabf185c6, 0xf5383f95, 0x696aee31, 0x887d8539, 0x3292b203, 0xb11edbc4, 0x4a252b18, 0x50c5a1f1, 0x2461fc2b, 0xafbe864c, 0xc8874c1a, 0xd5aa0734, 0x699c2550, 0x4ae8a6d, 0x8bf7373, 0xc0097529, 0x9fc06dd2, 0xfd26000f, 0xa91f6c40, 0xbf87c8c4, 0x4c6a3019, 0x47cb2d6f, 0x7235dee2, 0xca0e35d7, 0x38dd2787, 0x1a6d051a, 0x12c7fe99, 0x72a92668, 0x78200416, 0xe899b709, 0x189ebec3, 0x4b2a4be7, 0xe743447e, 0x49aba517, 0x1e96ed7d, 0x21498d64, 0x8659409b, 0x4f53ba5, 0x15685139, 0xadc82a53, 0xeab76ca6, 0x362d4288, 0xe9132520, 0x43fadda2, 0x4c86af0a, 0x7dd7bd17, 0x959a868f, 0xda92aef, 0x90df9c57, 0x92f95f19, 0x9d2b0fc7, 0x259000f3, 0xf4df855c, 0x96dcba4a, 0x42cf84be, 0xb3235d2e, 0x3b23d3a0, 0x1a3296d8, 0x88901497, 0x69ff92b7, 0xf32f2541, 0xb1c43571, 0x7e39274d, 0x6a07a703, 0x8a64cb80, 0xcc9bff8, 0xc3f6d4fd, 0x893081cd, 0xb9bc706, 0xa9ee7234, 0x23d4d1d5, 0x83cfc5c6, 0xcadaf602, 0xf1d0f64c, 0x7a10fdb, 0x96287895, 0xe21482cb, 0xe7455907, 0x8a71184d, 0x23318ac8, 0x72aed44b, 0x23a79d4c, 0xe463042b, 0xceb13214, 0x60a95e1e, 0x65ce2624, 0x89d63367, 0x2a54a707, 0xa6fd0f2d, 0xed7133c4, 0x5c7b9a39, 0x59072ef0, 0x922d9dfc, 0xc03538d2, 0xa3494060, 0xb9db010d, 0x20562c0a, 0xe2205917, 0xb0b13f7c, 0x9fa8fb51, 0xa5d2e787, 0xc03dc379, 0x5a9bc098, 0x595166ca, 0xc3633ddc, 0x4519fe94, 0x5b45a9a0, 0xe558cc8c, 0xc0b6bcc1, 0x6d97625e, 0xe19f9e3c, 0xf6ffc1e9, 0x2fcc098, 0xa9d74d52, 0x7f841373, 0x9f277361, 0x12e40882, 0x1d5ffdec, 0xc975a6dc, 0xf311ba67, 0x1066b762, 0x732d710a, 0x5af59eb6, 0x94115760, 0xf11c39e9, 0x687ba752, 0x613ac9d6, 0x3cadccc6, 0xc34ebac2, 0xe743e147, 0xc5873fe7, 0x92dca8b0, 0x4d25f636, 0xbc1c0f, 0xc5d23b85, 0x9dfd3591, 0x272668aa, 0x539f0014, 0x94215a76, 0x86ed47b0, 0x24e5916, 0xe2cd19e5, 0xb583cf8d, 0x5b7607f1, 0x787bea72, 0xe895c261, 0xc3bc56b5, 0x9f948754, 0x77a70c0a, 0x40cdc60, 0x44dd40b7, 0xedef993d, 0xd4e9ce38, 0xb0de9ed9, 0x8d23a195, 0xff508ce7, 0x11f0dc60, 0x2c1e81f9, 0x78f41fd3, 0x231b5419, 0xbe25d5f1, 0xeebe938e, 0x3124088d, 0xb262acd4, 0x76dd1e74, 0x10e5632b, 0x3af90e86, 0xc16606cd, 0x821c82b8, 0xc0fe07bb, 0x3564bb8f, 0xec4b5618, 0xc9f33fb, 0xb6267c89, 0x84d421c2, 0x1fd038c7, 0x2c1a6f35, 0x516dd45, 0x6574c9a4, 0x6b5c365, 0x1b7c4af8, 0x73f1815, 0x8282ed28, 0x3f07b04d, 0x1a2588db, 0xdc2acab1, 0x49e96971, 0x89f0d52f, 0x3b5485cd, 0x8d85bda8, 0xf7b915ce, 0xd78f562d, 0x473fe757, 0x1fc9d290, 0x3bceaf9c, 0x477894ea, 0x17703e2d, 0x95f451c3, 0x91efe374, 0xf8376d42, 0x6afd3bce, 0x8f9fad2e, 0x5e2af980, 0x4c6119d, 0xd020703f, 0xccf6434b, 0x4a2db623, 0x3ba433b5, 0xb7a6f9dd, 0xce9e7221, 0x9cea2a82, 0x634ad5d4, 0x6d49f653, 0xdd111430, 0xc07abf36, 0xbf435d20, 0x6d8520fd, 0x8e65c29c, 0x6daa7837, 0x22eebad2, 0x5ca5c51f, 0xf56c41c, 0x274130de, 0x1f105d4a, 0xefebb488, 0xb67c06d, 0xea99d663, 0x1b84bc20, 0xc82e6e08, 0x39c81654, 0x9dc7bbcd, 0xb6867232, 0x937424f, 0x8f499ca9, 0xa781c340, 0x336de6c, 0x213894d3, 0x126ce818, 0x4b5b2471, 0xf79f6bea, 0x5c5e5b14, 0x916f082c, 0x464f250f, 0x340b069e, 0x12efa1db, 0x40986b8e, 0x26f275f3, 0xbe6b5f13, 0x295fd4fb, 0x3207aa9a, 0xf04f68eb, 0x94d3f056, 0x5f02a187, 0xf8533ae3, 0xccdfc35, 0xd8c9c5af, 0xbdaf436e, 0x3d66a9bc, 0x4c764ef3, 0x7e67b122, 0x3450d064, 0x9eb56ae4, 0xfb546720, 0xd4381793, 0xe49d09c3, 0x28237b18, 0xc264aed3, 0x4c168e5, 0xa6afd65e, 0x11ecf8a6, 0xa28b2af, 0x7c811ce7, 0xd05a3385, 0x9b38684f, 0xc337fe3f, 0x91a03f71, 0x748d6457, 0x513ea9a0, 0x8825c72a, 0xfd15f2ca, 0x1bfd72c2, 0x946a2ed8, 0x256266e2, 0x6150ee9e, 0x823ffa91, 0x8d0af144, 0x3734db40, 0xbece3cbc, 0xea9c0df4, 0xab52d8ce, 0x764fa091, 0x43d1c4a7, 0x21dc44a3, 0x10fb51d1, 0xc3a5569c, 0x5ebdd696, 0x3675b726, 0xa1352aa9, 0x130299ca, 0x35cdf6cd, 0x186d6546, 0xc0b49699, 0x29864ce6, 0x110888f0, 0x5b413430, 0x42a402b2, 0x186617be, 0xce04652c, 0x24ace269, 0x3184e8b6, 0xee56dd00, 0xa3b3b556, 0x939b05f7, 0x8650cb10, 0xd6f2fd13, 0xecc03709, 0x9f901270, 0x43676adc, 0x53139433, 0x10e2d36f, 0xba5ef88a, 0xbc2d487b, 0x85d21144, 0xc5ad7597, 0xbc9fad0e, 0xe8669ebf, 0x2a580e15, 0xee95b8ae, 0xafe30d7c, 0x3925483, 0x6d4379b8, 0x3bff2758, 0xba82bcd0, 0x9de52fae, 0xc19f3236, 0xf2f2bbcf, 0x65c9da47, 0xf33ebdfc, 0xecd9ea0e, 0x8e80d2cd, 0x3419408f, 0xea652d99, 0x20c7735, 0xa43daed4, 0xed2551f9, 0x63d78bba, 0x4b651bd5, 0x7c69e272, 0x2abd56ee, 0x9ab4179d, 0x62b862c, 0x8caf2c25, 0x73b66096, 0xed1a39db, 0xceef49b6, 0xeb321d4c, 0x5e4df20c, 0x65169178, 0x9bf4fd3b, 0xf6985f9a, 0x8ebc7cc, 0x2c885bc7, 0x5ac592d7, 0x205721a7, 0x1419e27d, 0x22948c2b, 0xb17495cc, 0x81731372, 0x3412326, 0x582a65b, 0x75a4c270, 0xf2ae2cfd, 0xf618e345, 0xd3bdd6e2, 0x558d29a1, 0x3d848ab, 0x78da759a, 0x2d1b32b8, 0x1afaebc5, 0x55021e49, 0x80c99534, 0x21890259, 0xe2bc921b, 0xcf3ac82b, 0x88ccf401, 0x5840a952, 0x480edc7f, 0xf0aae25e, 0x5ac60fe0, 0x94fee19a, 0xe5729f54, 0xe0fb221e, 0x3dd23b67, 0xd8408440, 0x61d4496, 0xe7c7ca62, 0xf739df34, 0x75bab683, 0x6e126046, 0x8bdd628c, 0x58a65d27, 0xcc7148e6, 0x93bab453, 0x4924deef, 0x206404b0, 0x7d81413d, 0xf33609c6, 0x995f4b0d, 0x5048cca4, 0x3f9fd17, 0xf3ecbd96, 0x97ee65c2, 0x37e89e7a, 0x6f0569dc, 0x3faad0f9, 0xceaf2dea, 0xdd221d5f, 0x50b3e235, 0x3c3a8b89, 0xe4945aa9, 0xd0aa65b9, 0x93ee7358, 0x8b3cd96d, 0x2f1b4af6, 0x30f66edd, 0xc9b5baf4, 0x96be36cc, 0x9cae1352, 0xc52b16a, 0xdccff8b, 0x4104069, 0x6b91d024, 0xcb9dc70, 0xadd77490, 0x6640149e, 0xeb297c73, 0x93affa6a, 0x1a58d4, 0xde159fb9, 0xfa0ce323, 0xc941eda9, 0x60679151, 0x41343302, 0xf949e0d5, 0x150ea85d, 0x9aceaca4, 0x2862b26d, 0xd42f3d2e, 0x35a15350, 0x93204b1b, 0x6ad99948, 0xa0c99a1c, 0x59023682, 0x491b86e3, 0xb32b998d, 0x963ab5a9, 0xb28ffb79, 0xc0016d24, 0x13211ae4, 0xdbba74a8, 0x9561412, 0xc14d10e1, 0xca161ba6, 0xb2b3e13f, 0xe68909f3, 0xdd4e862e, 0x1319d68, 0x529b38d9, 0x43a2d3e7, 0xabb8d070, 0x77a609ff, 0x736c7965, 0x5c1cea02, 0x61d17c62, 0x3d7ee71c, 0x692aeece, 0x5ee62cb, 0x66c75dba, 0x881b66aa, 0x5140131a, 0x215470e8, 0x9f361b95, 0x8fbbc63a, 0x6e24af7f, 0x2af6f590, 0xf94b17dc, 0x6bcb8e25, 0xad845be9, 0xaf47b4ed, 0x32d579e8, 0x461e54ea, 0x6d3c470b, 0x28f5796, 0x57e58a06, 0x38492860, 0xcc320e8e, 0x82e4864, 0xe1479157, 0xef53c8cb, 0xe7622fa3, 0xc8372ebf, 0xa9a8017b, 0x84810a20, 0x452c5e6e, 0x436e75b, 0x409b1ab1, 0x4c7e2be6, 0xdada977a, 0x8a6c9880, 0x87184e96, 0xdc215461, 0xcd5a6dc6, 0x2f63a6a4, 0x928e9b92, 0x15634b1b, 0xbbaf3a9c, 0xb556de5, 0x84ddf234, 0x719d6666, 0xc558a5f2, 0x4f212de, 0x91a0ad79, 0x11290332, 0x7738c32e, 0xcfd04207, 0x57ba75d5, 0xa2e3f110, 0x11762ae7, 0xc5d9869d, 0x60bfa38b, 0xb4a795bd, 0x14625ffb, 0x3cfb63fa, 0xfb99e70e, 0xb5ef968, 0x2e7e2e1e, 0x47fdd488, 0xcfd5f8a8, 0xdd7683b0, 0xdffd7c9e, 0x12b83597, 0xb03bd9c2, 0xa90b82dc, 0x91ca62c0, 0x2ff80ae7, 0x29356c8c, 0x85b49291, 0x778572fa, 0xe764a1a1, 0x585d320a, 0x79a39c1c, 0x399c37e9, 0xd8d673cb, 0x97aedb85, 0xd6ba3f51, 0x4ff0ea85, 0x23a7db79, 0xea93002a, 0x8a516656, 0xe8dddfb4, 0x7792e739, 0x41d28138, 0x37fb9a4e, 0x1c635f57, 0x70da1ffa, 0x31660175, 0x7af382d1, 0x7fe4bf8a, 0x55521ec8, 0xba849730, 0x24819f5, 0x354c3b13, 0x8166158c, 0x3f7f2099, 0xac53614d, 0xda072b10, 0xe329504, 0x6a750fd8, 0x28572376, 0x9ddf129b, 0x9ae80bb2, 0x3bd18b44, 0x51b8a683, 0x1a1a86ba, 0xe3f8ee0c, 0x840f3482, 0xca55eb4c, 0x7a25e960, 0x37ae0760, 0x27157d3d, 0x3340e58f, 0x9f2eb164, 0xae3151fa, 0x8b443f4a, 0x27fdd38c, 0xa7758b92, 0x82a7049e, 0x2500efbb, 0xfcf1ce85, 0xc0642090, 0x250a875f, 0x38d83660, 0xc5d63bc, 0x84f43dab, 0xe6290eec, 0xc909291f, 0xb71fd6eb, 0x5b77263, 0x4f71a52c, 0x53096b18, 0x6497bbbe, 0xdf77d74f, 0x14ae3e15, 0xd83ee393, 0x5a249f4d, 0x89d73e21, 0x426804, 0xddda0a27, 0x9fe7c87c, 0xf3267c43, 0x3344fca2, 0xd38f68a2, 0xa007502a, 0xdaa74f4e, 0x524aaad2, 0x19473f66, 0xc1d0c26, 0xa6bbe17a, 0xb92e6eb0, 0xb419b07a, 0xc25e56d1, 0x9c38bdfe, 0x68038f05, 0xccb395fa, 0xb44f4ce6, 0x8d9a8db, 0x95da19f5, 0xc52e5d88, 0xf8356128, 0xbb5290c5, 0x4dcafa05, 0x427b9807, 0x505245ce, 0x41cf4b25, 0x39f96369, 0xa1dea14e, 0xac1e1e21, 0x58656a1b, 0xc76772b3, 0xcbed3c4d, 0x833dcc1e, 0x7236a3ab, 0x9d8200ae, 0xc8624409, 0xbb07343, 0xfd8f8f03, 0x723393c3, 0x4cdd1336, 0xf7cfad61, 0x249c07bf, 0x5c0c24e3, 0xe6bc2650, 0x8dd50d1c, 0x8aa3a2a5, 0x694e85c0, 0xf98895a5, 0xf74ed3dd, 0xa2f88192, 0x33fb994e, 0xfe711567, 0xfe29e1c6, 0x8bcb4bdc, 0xa8c15434, 0x86c3d98a, 0xcf1fafa0, 0x22ab672e, 0x7549603f, 0x5b0f7e7b, 0xe3269edd, 0x6b64292, 0x88afeba4, 0x290fa89d, 0x93c895f8, 0xbee21427, 0xeae12b1c, 0x7c84636, 0x506be81d, 0x5dd5c4b7, 0xb9e43567, 0xdcc2b95a, 0xce23ef9b, 0xb15353bc, 0x7ae5676a, 0xb0e1958c, 0x7cd1ac91, 0x304a8047, 0xd766a65b, 0x7120a387, 0x8e370d7c, 0x94e241d3, 0x4516f0b8, 0xfd605e33, 0x7c9b2ba0, 0x343332cc, 0x469dd63e, 0x3f6b6c66, 0x39b330b3, 0x431dc660, 0xa591aa1e, 0xc00b4c96, 0xebcf6905, 0x74fc55db, 0x3352505f, 0xe92ed69, 0xe13e0d96, 0x822e1ecc, 0x773f2c74, 0x3643056d, 0xb9984c16, 0xcc7153e6, 0x4d88f992, 0x4c1de8dd, 0x7d2f951d, 0x7113012, 0x34240034, 0x97eb307a, 0x16af8e1f, 0xd805ed31, 0x2a6aa2e2, 0x618a45d3, 0x80c46179, 0xbff6b6dd, 0xc99ee8ad, 0x82da2aeb, 0xe75273b3, 0x8a7bd156, 0x3832bb0f, 0xf59d7cc7, 0xbcf626b0, 0xcdd0619c, 0x41ff5ed3, 0x846531b, 0x906ffbfd, 0xb59a672b, 0x21e6e893, 0x770a564f, 0x5f412346, 0xf2929158, 0x153603fa, 0x38afd0d6, 0x8177460b, 0x445ed4e5, 0xb1c4a5c1, 0x14db7a5d, 0x74fc810f, 0x6db9f6b3, 0x5c7c7fe2, 0x1be8a756, 0xd0e5de2d, 0xa23fd973, 0xbba4dbf, 0xcd8effaa, 0x6348ae96, 0xb261846d, 0x7561c667, 0xc4266dd2, 0xbc58f0a8, 0x57ab12ac, 0xe8ddf790, 0xd889b911, 0x47040b6b, 0x6dc3ca58, 0xd15944e6, 0xd2f24f80, 0xa80450f0, 0xa0620d61, 0x2e34452, 0x24b75f77, 0xbc4023c7, 0x141129f4, 0x9a7ea00e, 0x4b172a4, 0xae15bc1a, 0x1114e3a0, 0x957e6381, 0x7565bfb0, 0x92d24558, 0x1d03f9f5, 0x5c6ba416, 0x71ccf22, 0xf56d5b5e, 0xc13e9a15, 0xfca848c4, 0x6514ebc3, 0x631b4eb0, 0xbf375c76, 0xd60fe492, 0x73d0cd61, 0xfb6dae8b, 0x7338e28b, 0x95d9b179, 0x7a62f5b4, 0xabc2334c, 0x79585244, 0x1614e8ad, 0xcd9c1cc0, 0xfdcfb459, 0x67033228, 0xc78d5307, 0xe7998307, 0xfe8eb0f6, 0x97cd521, 0xa758363a, 0xc61ca107, 0x994e0a2, 0x202a0884, 0xb8163d85, 0x9e56e7bd, 0xb0286076, 0x2a73b67, 0x9bdb2cb2, 0x89e37a4c, 0x1b30a9ad, 0xc5c8fe, 0x47daf560, 0xf3816f95, 0xc6b60072, 0xe7c86d7d, 0xe3e1f05c, 0xcbc47d47, 0x82b6a2dc, 0xea4f6aef, 0xb738a299, 0x25434de9, 0x13e950f7, 0x2860ff61, 0x9d849401, 0x3008a10d, 0xd67a5074, 0x9f5be426, 0xb22d68d5, 0xe7e3219b, 0x8e185ea6, 0xfd6db3d4, 0xdc76eca1, 0xb60c1f01, 0xbecd63fe, 0xbb57459f, 0x43b980f5, 0xe8c77071, 0xfff0de72, 0x669fa926, 0x92e0181c, 0x3ff636cb, 0xa3995e72, 0x2c65eb63, 0x7d5cf41f, 0x1e94bcb3, 0xe3ae1042, 0xd0073f43, 0xdfdfe7a0, 0x25942f4a, 0xd908c3a0, 0x43a8ffe1, 0xa5d13b05, 0xd1afd400, 0x95e87ba5, 0x4f82a56e, 0x4815cf48, 0xfb7fb5ab, 0xfddd096c, 0x44426f18, 0xccb69890, 0x889d817d, 0x806b3b60, 0x5080e8e0, 0xc5b67529, 0xe9286898, 0x1fd0d115, 0x5dd6a7f9, 0xebc7d39b, 0x6effe82e, 0xb7a6923, 0x83259d3e, 0x3a85d844, 0xf05add3e, 0xd7ad2f7d, 0x7ec4620, 0x4e39f0f9, 0xb7855c2d, 0x7bfc7f58, 0xe419d166, 0xa669e973, 0x6fc4baa, 0x24d97039, 0x85a52699, 0x4e48cb20, 0x53741168, 0xe30d3821, 0xdc077df6, 0xe3081286, 0x8efaf0c1, 0x83e53d33, 0xb0b2c61f, 0x445fd987, 0x73ee34fb, 0xe15e308d, 0xa0d8dc7f, 0x7f7bd583, 0x4a43c99c, 0x555c468f, 0x264ab1d, 0xe50ad642, 0x93a646ef, 0x3459003d, 0x4fbaced3, 0xd1d19c60, 0x846ba63a, 0xd6be13d2, 0xea9992e8, 0x85d5f5fa, 0x6d2d73a2, 0x2974c31e, 0x3f5558b9, 0xf2f02860, 0x5f0d1bb1, 0xf380dfc, 0xee8e732e, 0xe2f1a153, 0xefd69b29, 0x93083914, 0xd82601f8, 0x60fc1c28, 0xeb92a778, 0xa4125dfe, 0x3a57ac6e, 0x20b7f5b2], 1);
    test(0x1, [0xb8672f8c, 0xeebc1448, 0x77eff, 0x20ccc389, 0x4d65aacb, 0xffc11e85, 0x2591cb4f, 0x3c7053c0, 0x17a38090, 0x65865081, 0x2faebfcc, 0x634e1e47, 0x5876aaed, 0xab7479fc, 0x65928d86, 0xef7f7d19, 0x89efe4b2, 0xd8a7d514, 0x6b5054fe, 0x5032b165, 0xaf6a8b92, 0x8648c9d4, 0x3456fb8b, 0x718620fc, 0xe0cc4dea, 0x3ac5929c, 0x702df9d, 0x88cf598e, 0xaba3c232, 0xe9f96a44, 0x6ad47cd7, 0x750b5fd7, 0x8f064be9, 0x6e4242f1, 0x23f065f7, 0xf06aadde, 0x32b6c760, 0xc7448456, 0xccfd988d, 0xb749d7e9, 0xf7dffd89, 0xcd818407, 0x503c913f, 0x17c1cb3d, 0xb13c0e16, 0x84a5a539, 0xe05b0a01, 0xdd71f780, 0xe504f13c, 0xd442f8d1, 0x15c57508, 0xd460e2d8, 0x9ff7e8d, 0x45e69a73, 0x2b7a01af, 0xf2a8fea, 0xe0cdf279, 0xaba7b9c8, 0x192d421e, 0x97d32447, 0x6bcdb583, 0xabf185c6, 0xf5383f95, 0x696aee31, 0x887d8539, 0x3292b203, 0xb11edbc4, 0x4a252b18, 0x50c5a1f1, 0x2461fc2b, 0xafbe864c, 0xc8874c1a, 0xd5aa0734, 0x699c2550, 0x4ae8a6d, 0x8bf7373, 0xc0097529, 0x9fc06dd2, 0xfd26000f, 0xa91f6c40, 0xbf87c8c4, 0x4c6a3019, 0x47cb2d6f, 0x7235dee2, 0xca0e35d7, 0x38dd2787, 0x1a6d051a, 0x12c7fe99, 0x72a92668, 0x78200416, 0xe899b709, 0x189ebec3, 0x4b2a4be7, 0xe743447e, 0x49aba517, 0x1e96ed7d, 0x21498d64, 0x8659409b, 0x4f53ba5, 0x15685139, 0xadc82a53, 0xeab76ca6, 0x362d4288, 0xe9132520, 0x43fadda2, 0x4c86af0a, 0x7dd7bd17, 0x959a868f, 0xda92aef, 0x90df9c57, 0x92f95f19, 0x9d2b0fc7, 0x259000f3, 0xf4df855c, 0x96dcba4a, 0x42cf84be, 0xb3235d2e, 0x3b23d3a0, 0x1a3296d8, 0x88901497, 0x69ff92b7, 0xf32f2541, 0xb1c43571, 0x7e39274d, 0x6a07a703, 0x8a64cb80, 0xcc9bff8, 0xc3f6d4fd, 0x893081cd, 0xb9bc706, 0xa9ee7234, 0x23d4d1d5, 0x83cfc5c6, 0xcadaf602, 0xf1d0f64c, 0x7a10fdb, 0x96287895, 0xe21482cb, 0xe7455907, 0x8a71184d, 0x23318ac8, 0x72aed44b, 0x23a79d4c, 0xe463042b, 0xceb13214, 0x60a95e1e, 0x65ce2624, 0x89d63367, 0x2a54a707, 0xa6fd0f2d, 0xed7133c4, 0x5c7b9a39, 0x59072ef0, 0x922d9dfc, 0xc03538d2, 0xa3494060, 0xb9db010d, 0x20562c0a, 0xe2205917, 0xb0b13f7c, 0x9fa8fb51, 0xa5d2e787, 0xc03dc379, 0x5a9bc098, 0x595166ca, 0xc3633ddc, 0x4519fe94, 0x5b45a9a0, 0xe558cc8c, 0xc0b6bcc1, 0x6d97625e, 0xe19f9e3c, 0xf6ffc1e9, 0x2fcc098, 0xa9d74d52, 0x7f841373, 0x9f277361, 0x12e40882, 0x1d5ffdec, 0xc975a6dc, 0xf311ba67, 0x1066b762, 0x732d710a, 0x5af59eb6, 0x94115760, 0xf11c39e9, 0x687ba752, 0x613ac9d6, 0x3cadccc6, 0xc34ebac2, 0xe743e147, 0xc5873fe7, 0x92dca8b0, 0x4d25f636, 0xbc1c0f, 0xc5d23b85, 0x9dfd3591, 0x272668aa, 0x539f0014, 0x94215a76, 0x86ed47b0, 0x24e5916, 0xe2cd19e5, 0xb583cf8d, 0x5b7607f1, 0x787bea72, 0xe895c261, 0xc3bc56b5, 0x9f948754, 0x77a70c0a, 0x40cdc60, 0x44dd40b7, 0xedef993d, 0xd4e9ce38, 0xb0de9ed9, 0x8d23a195, 0xff508ce7, 0x11f0dc60, 0x2c1e81f9, 0x78f41fd3, 0x231b5419, 0xbe25d5f1, 0xeebe938e, 0x3124088d, 0xb262acd4, 0x76dd1e74, 0x10e5632b, 0x3af90e86, 0xc16606cd, 0x821c82b8, 0xc0fe07bb, 0x3564bb8f, 0xec4b5618, 0xc9f33fb, 0xb6267c89, 0x84d421c2, 0x1fd038c7, 0x2c1a6f35, 0x516dd45, 0x6574c9a4, 0x6b5c365, 0x1b7c4af8, 0x73f1815, 0x8282ed28, 0x3f07b04d, 0x1a2588db, 0xdc2acab1, 0x49e96971, 0x89f0d52f, 0x3b5485cd, 0x8d85bda8, 0xf7b915ce, 0xd78f562d, 0x473fe757, 0x1fc9d290, 0x3bceaf9c, 0x477894ea, 0x17703e2d, 0x95f451c3, 0x91efe374, 0xf8376d42, 0x6afd3bce, 0x8f9fad2e, 0x5e2af980, 0x4c6119d, 0xd020703f, 0xccf6434b, 0x4a2db623, 0x3ba433b5, 0xb7a6f9dd, 0xce9e7221, 0x9cea2a82, 0x634ad5d4, 0x6d49f653, 0xdd111430, 0xc07abf36, 0xbf435d20, 0x6d8520fd, 0x8e65c29c, 0x6daa7837, 0x22eebad2, 0x5ca5c51f, 0xf56c41c, 0x274130de, 0x1f105d4a, 0xefebb488, 0xb67c06d, 0xea99d663, 0x1b84bc20, 0xc82e6e08, 0x39c81654, 0x9dc7bbcd, 0xb6867232, 0x937424f, 0x8f499ca9, 0xa781c340, 0x336de6c, 0x213894d3, 0x126ce818, 0x4b5b2471, 0xf79f6bea, 0x5c5e5b14, 0x916f082c, 0x464f250f, 0x340b069e, 0x12efa1db, 0x40986b8e, 0x26f275f3, 0xbe6b5f13, 0x295fd4fb, 0x3207aa9a, 0xf04f68eb, 0x94d3f056, 0x5f02a187, 0xf8533ae3, 0xccdfc35, 0xd8c9c5af, 0xbdaf436e, 0x3d66a9bc, 0x4c764ef3, 0x7e67b122, 0x3450d064, 0x9eb56ae4, 0xfb546720, 0xd4381793, 0xe49d09c3, 0x28237b18, 0xc264aed3, 0x4c168e5, 0xa6afd65e, 0x11ecf8a6, 0xa28b2af, 0x7c811ce7, 0xd05a3385, 0x9b38684f, 0xc337fe3f, 0x91a03f71, 0x748d6457, 0x513ea9a0, 0x8825c72a, 0xfd15f2ca, 0x1bfd72c2, 0x946a2ed8, 0x256266e2, 0x6150ee9e, 0x823ffa91, 0x8d0af144, 0x3734db40, 0xbece3cbc, 0xea9c0df4, 0xab52d8ce, 0x764fa091, 0x43d1c4a7, 0x21dc44a3, 0x10fb51d1, 0xc3a5569c, 0x5ebdd696, 0x3675b726, 0xa1352aa9, 0x130299ca, 0x35cdf6cd, 0x186d6546, 0xc0b49699, 0x29864ce6, 0x110888f0, 0x5b413430, 0x42a402b2, 0x186617be, 0xce04652c, 0x24ace269, 0x3184e8b6, 0xee56dd00, 0xa3b3b556, 0x939b05f7, 0x8650cb10, 0xd6f2fd13, 0xecc03709, 0x9f901270, 0x43676adc, 0x53139433, 0x10e2d36f, 0xba5ef88a, 0xbc2d487b, 0x85d21144, 0xc5ad7597, 0xbc9fad0e, 0xe8669ebf, 0x2a580e15, 0xee95b8ae, 0xafe30d7c, 0x3925483, 0x6d4379b8, 0x3bff2758, 0xba82bcd0, 0x9de52fae, 0xc19f3236, 0xf2f2bbcf, 0x65c9da47, 0xf33ebdfc, 0xecd9ea0e, 0x8e80d2cd, 0x3419408f, 0xea652d99, 0x20c7735, 0xa43daed4, 0xed2551f9, 0x63d78bba, 0x4b651bd5, 0x7c69e272, 0x2abd56ee, 0x9ab4179d, 0x62b862c, 0x8caf2c25, 0x73b66096, 0xed1a39db, 0xceef49b6, 0xeb321d4c, 0x5e4df20c, 0x65169178, 0x9bf4fd3b, 0xf6985f9a, 0x8ebc7cc, 0x2c885bc7, 0x5ac592d7, 0x205721a7, 0x1419e27d, 0x22948c2b, 0xb17495cc, 0x81731372, 0x3412326, 0x582a65b, 0x75a4c270, 0xf2ae2cfd, 0xf618e345, 0xd3bdd6e2, 0x558d29a1, 0x3d848ab, 0x78da759a, 0x2d1b32b8, 0x1afaebc5, 0x55021e49, 0x80c99534, 0x21890259, 0xe2bc921b, 0xcf3ac82b, 0x88ccf401, 0x5840a952, 0x480edc7f, 0xf0aae25e, 0x5ac60fe0, 0x94fee19a, 0xe5729f54, 0xe0fb221e, 0x3dd23b67, 0xd8408440, 0x61d4496, 0xe7c7ca62, 0xf739df34, 0x75bab683, 0x6e126046, 0x8bdd628c, 0x58a65d27, 0xcc7148e6, 0x93bab453, 0x4924deef, 0x206404b0, 0x7d81413d, 0xf33609c6, 0x995f4b0d, 0x5048cca4, 0x3f9fd17, 0xf3ecbd96, 0x97ee65c2, 0x37e89e7a, 0x6f0569dc, 0x3faad0f9, 0xceaf2dea, 0xdd221d5f, 0x50b3e235, 0x3c3a8b89, 0xe4945aa9, 0xd0aa65b9, 0x93ee7358, 0x8b3cd96d, 0x2f1b4af6, 0x30f66edd, 0xc9b5baf4, 0x96be36cc, 0x9cae1352, 0xc52b16a, 0xdccff8b, 0x4104069, 0x6b91d024, 0xcb9dc70, 0xadd77490, 0x6640149e, 0xeb297c73, 0x93affa6a, 0x1a58d4, 0xde159fb9, 0xfa0ce323, 0xc941eda9, 0x60679151, 0x41343302, 0xf949e0d5, 0x150ea85d, 0x9aceaca4, 0x2862b26d, 0xd42f3d2e, 0x35a15350, 0x93204b1b, 0x6ad99948, 0xa0c99a1c, 0x59023682, 0x491b86e3, 0xb32b998d, 0x963ab5a9, 0xb28ffb79, 0xc0016d24, 0x13211ae4, 0xdbba74a8, 0x9561412, 0xc14d10e1, 0xca161ba6, 0xb2b3e13f, 0xe68909f3, 0xdd4e862e, 0x1319d68, 0x529b38d9, 0x43a2d3e7, 0xabb8d070, 0x77a609ff, 0x736c7965, 0x5c1cea02, 0x61d17c62, 0x3d7ee71c, 0x692aeece, 0x5ee62cb, 0x66c75dba, 0x881b66aa, 0x5140131a, 0x215470e8, 0x9f361b95, 0x8fbbc63a, 0x6e24af7f, 0x2af6f590, 0xf94b17dc, 0x6bcb8e25, 0xad845be9, 0xaf47b4ed, 0x32d579e8, 0x461e54ea, 0x6d3c470b, 0x28f5796, 0x57e58a06, 0x38492860, 0xcc320e8e, 0x82e4864, 0xe1479157, 0xef53c8cb, 0xe7622fa3, 0xc8372ebf, 0xa9a8017b, 0x84810a20, 0x452c5e6e, 0x436e75b, 0x409b1ab1, 0x4c7e2be6, 0xdada977a, 0x8a6c9880, 0x87184e96, 0xdc215461, 0xcd5a6dc6, 0x2f63a6a4, 0x928e9b92, 0x15634b1b, 0xbbaf3a9c, 0xb556de5, 0x84ddf234, 0x719d6666, 0xc558a5f2, 0x4f212de, 0x91a0ad79, 0x11290332, 0x7738c32e, 0xcfd04207, 0x57ba75d5, 0xa2e3f110, 0x11762ae7, 0xc5d9869d, 0x60bfa38b, 0xb4a795bd, 0x14625ffb, 0x3cfb63fa, 0xfb99e70e, 0xb5ef968, 0x2e7e2e1e, 0x47fdd488, 0xcfd5f8a8, 0xdd7683b0, 0xdffd7c9e, 0x12b83597, 0xb03bd9c2, 0xa90b82dc, 0x91ca62c0, 0x2ff80ae7, 0x29356c8c, 0x85b49291, 0x778572fa, 0xe764a1a1, 0x585d320a, 0x79a39c1c, 0x399c37e9, 0xd8d673cb, 0x97aedb85, 0xd6ba3f51, 0x4ff0ea85, 0x23a7db79, 0xea93002a, 0x8a516656, 0xe8dddfb4, 0x7792e739, 0x41d28138, 0x37fb9a4e, 0x1c635f57, 0x70da1ffa, 0x31660175, 0x7af382d1, 0x7fe4bf8a, 0x55521ec8, 0xba849730, 0x24819f5, 0x354c3b13, 0x8166158c, 0x3f7f2099, 0xac53614d, 0xda072b10, 0xe329504, 0x6a750fd8, 0x28572376, 0x9ddf129b, 0x9ae80bb2, 0x3bd18b44, 0x51b8a683, 0x1a1a86ba, 0xe3f8ee0c, 0x840f3482, 0xca55eb4c, 0x7a25e960, 0x37ae0760, 0x27157d3d, 0x3340e58f, 0x9f2eb164, 0xae3151fa, 0x8b443f4a, 0x27fdd38c, 0xa7758b92, 0x82a7049e, 0x2500efbb, 0xfcf1ce85, 0xc0642090, 0x250a875f, 0x38d83660, 0xc5d63bc, 0x84f43dab, 0xe6290eec, 0xc909291f, 0xb71fd6eb, 0x5b77263, 0x4f71a52c, 0x53096b18, 0x6497bbbe, 0xdf77d74f, 0x14ae3e15, 0xd83ee393, 0x5a249f4d, 0x89d73e21, 0x426804, 0xddda0a27, 0x9fe7c87c, 0xf3267c43, 0x3344fca2, 0xd38f68a2, 0xa007502a, 0xdaa74f4e, 0x524aaad2, 0x19473f66, 0xc1d0c26, 0xa6bbe17a, 0xb92e6eb0, 0xb419b07a, 0xc25e56d1, 0x9c38bdfe, 0x68038f05, 0xccb395fa, 0xb44f4ce6, 0x8d9a8db, 0x95da19f5, 0xc52e5d88, 0xf8356128, 0xbb5290c5, 0x4dcafa05, 0x427b9807, 0x505245ce, 0x41cf4b25, 0x39f96369, 0xa1dea14e, 0xac1e1e21, 0x58656a1b, 0xc76772b3, 0xcbed3c4d, 0x833dcc1e, 0x7236a3ab, 0x9d8200ae, 0xc8624409, 0xbb07343, 0xfd8f8f03, 0x723393c3, 0x4cdd1336, 0xf7cfad61, 0x249c07bf, 0x5c0c24e3, 0xe6bc2650, 0x8dd50d1c, 0x8aa3a2a5, 0x694e85c0, 0xf98895a5, 0xf74ed3dd, 0xa2f88192, 0x33fb994e, 0xfe711567, 0xfe29e1c6, 0x8bcb4bdc, 0xa8c15434, 0x86c3d98a, 0xcf1fafa0, 0x22ab672e, 0x7549603f, 0x5b0f7e7b, 0xe3269edd, 0x6b64292, 0x88afeba4, 0x290fa89d, 0x93c895f8, 0xbee21427, 0xeae12b1c, 0x7c84636, 0x506be81d, 0x5dd5c4b7, 0xb9e43567, 0xdcc2b95a, 0xce23ef9b, 0xb15353bc, 0x7ae5676a, 0xb0e1958c, 0x7cd1ac91, 0x304a8047, 0xd766a65b, 0x7120a387, 0x8e370d7c, 0x94e241d3, 0x4516f0b8, 0xfd605e33, 0x7c9b2ba0, 0x343332cc, 0x469dd63e, 0x3f6b6c66, 0x39b330b3, 0x431dc660, 0xa591aa1e, 0xc00b4c96, 0xebcf6905, 0x74fc55db, 0x3352505f, 0xe92ed69, 0xe13e0d96, 0x822e1ecc, 0x773f2c74, 0x3643056d, 0xb9984c16, 0xcc7153e6, 0x4d88f992, 0x4c1de8dd, 0x7d2f951d, 0x7113012, 0x34240034, 0x97eb307a, 0x16af8e1f, 0xd805ed31, 0x2a6aa2e2, 0x618a45d3, 0x80c46179, 0xbff6b6dd, 0xc99ee8ad, 0x82da2aeb, 0xe75273b3, 0x8a7bd156, 0x3832bb0f, 0xf59d7cc7, 0xbcf626b0, 0xcdd0619c, 0x41ff5ed3, 0x846531b, 0x906ffbfd, 0xb59a672b, 0x21e6e893, 0x770a564f, 0x5f412346, 0xf2929158, 0x153603fa, 0x38afd0d6, 0x8177460b, 0x445ed4e5, 0xb1c4a5c1, 0x14db7a5d, 0x74fc810f, 0x6db9f6b3, 0x5c7c7fe2, 0x1be8a756, 0xd0e5de2d, 0xa23fd973, 0xbba4dbf, 0xcd8effaa, 0x6348ae96, 0xb261846d, 0x7561c667, 0xc4266dd2, 0xbc58f0a8, 0x57ab12ac, 0xe8ddf790, 0xd889b911, 0x47040b6b, 0x6dc3ca58, 0xd15944e6, 0xd2f24f80, 0xa80450f0, 0xa0620d61, 0x2e34452, 0x24b75f77, 0xbc4023c7, 0x141129f4, 0x9a7ea00e, 0x4b172a4, 0xae15bc1a, 0x1114e3a0, 0x957e6381, 0x7565bfb0, 0x92d24558, 0x1d03f9f5, 0x5c6ba416, 0x71ccf22, 0xf56d5b5e, 0xc13e9a15, 0xfca848c4, 0x6514ebc3, 0x631b4eb0, 0xbf375c76, 0xd60fe492, 0x73d0cd61, 0xfb6dae8b, 0x7338e28b, 0x95d9b179, 0x7a62f5b4, 0xabc2334c, 0x79585244, 0x1614e8ad, 0xcd9c1cc0, 0xfdcfb459, 0x67033228, 0xc78d5307, 0xe7998307, 0xfe8eb0f6, 0x97cd521, 0xa758363a, 0xc61ca107, 0x994e0a2, 0x202a0884, 0xb8163d85, 0x9e56e7bd, 0xb0286076, 0x2a73b67, 0x9bdb2cb2, 0x89e37a4c, 0x1b30a9ad, 0xc5c8fe, 0x47daf560, 0xf3816f95, 0xc6b60072, 0xe7c86d7d, 0xe3e1f05c, 0xcbc47d47, 0x82b6a2dc, 0xea4f6aef, 0xb738a299, 0x25434de9, 0x13e950f7, 0x2860ff61, 0x9d849401, 0x3008a10d, 0xd67a5074, 0x9f5be426, 0xb22d68d5, 0xe7e3219b, 0x8e185ea6, 0xfd6db3d4, 0xdc76eca1, 0xb60c1f01, 0xbecd63fe, 0xbb57459f, 0x43b980f5, 0xe8c77071, 0xfff0de72, 0x669fa926, 0x92e0181c, 0x3ff636cb, 0xa3995e72, 0x2c65eb63, 0x7d5cf41f, 0x1e94bcb3, 0xe3ae1042, 0xd0073f43, 0xdfdfe7a0, 0x25942f4a, 0xd908c3a0, 0x43a8ffe1, 0xa5d13b05, 0xd1afd400, 0x95e87ba5, 0x4f82a56e, 0x4815cf48, 0xfb7fb5ab, 0xfddd096c, 0x44426f18, 0xccb69890, 0x889d817d, 0x806b3b60, 0x5080e8e0, 0xc5b67529, 0xe9286898, 0x1fd0d115, 0x5dd6a7f9, 0xebc7d39b, 0x6effe82e, 0xb7a6923, 0x83259d3e, 0x3a85d844, 0xf05add3e, 0xd7ad2f7d, 0x7ec4620, 0x4e39f0f9, 0xb7855c2d, 0x7bfc7f58, 0xe419d166, 0xa669e973, 0x6fc4baa, 0x24d97039, 0x85a52699, 0x4e48cb20, 0x53741168, 0xe30d3821, 0xdc077df6, 0xe3081286, 0x8efaf0c1, 0x83e53d33, 0xb0b2c61f, 0x445fd987, 0x73ee34fb, 0xe15e308d, 0xa0d8dc7f, 0x7f7bd583, 0x4a43c99c, 0x555c468f, 0x264ab1d, 0xe50ad642, 0x93a646ef, 0x3459003d, 0x4fbaced3, 0xd1d19c60, 0x846ba63a, 0xd6be13d2, 0xea9992e8, 0x85d5f5fa, 0x6d2d73a2, 0x2974c31e, 0x3f5558b9, 0xf2f02860, 0x5f0d1bb1, 0xf380dfc, 0xee8e732e, 0xe2f1a153, 0xefd69b29, 0x93083914, 0xd82601f8, 0x60fc1c28, 0xeb92a778, 0xa4125dfe, 0x3a57ac6e, 0x20b7f5b2], 2);
    test(0x1, [0xeebc1448, 0x77eff, 0x20ccc389, 0x4d65aacb, 0xffc11e85, 0x2591cb4f, 0x3c7053c0, 0x17a38090, 0x65865081, 0x2faebfcc, 0x634e1e47, 0x5876aaed, 0xab7479fc, 0x65928d86, 0xef7f7d19, 0x89efe4b2, 0xd8a7d514, 0x6b5054fe, 0x5032b165, 0xaf6a8b92, 0x8648c9d4, 0x3456fb8b, 0x718620fc, 0xe0cc4dea, 0x3ac5929c, 0x702df9d, 0x88cf598e, 0xaba3c232, 0xe9f96a44, 0x6ad47cd7, 0x750b5fd7, 0x8f064be9, 0x6e4242f1, 0x23f065f7, 0xf06aadde, 0x32b6c760, 0xc7448456, 0xccfd988d, 0xb749d7e9, 0xf7dffd89, 0xcd818407, 0x503c913f, 0x17c1cb3d, 0xb13c0e16, 0x84a5a539, 0xe05b0a01, 0xdd71f780, 0xe504f13c, 0xd442f8d1, 0x15c57508, 0xd460e2d8, 0x9ff7e8d, 0x45e69a73, 0x2b7a01af, 0xf2a8fea, 0xe0cdf279, 0xaba7b9c8, 0x192d421e, 0x97d32447, 0x6bcdb583, 0xabf185c6, 0xf5383f95, 0x696aee31, 0x887d8539, 0x3292b203, 0xb11edbc4, 0x4a252b18, 0x50c5a1f1, 0x2461fc2b, 0xafbe864c, 0xc8874c1a, 0xd5aa0734, 0x699c2550, 0x4ae8a6d, 0x8bf7373, 0xc0097529, 0x9fc06dd2, 0xfd26000f, 0xa91f6c40, 0xbf87c8c4, 0x4c6a3019, 0x47cb2d6f, 0x7235dee2, 0xca0e35d7, 0x38dd2787, 0x1a6d051a, 0x12c7fe99, 0x72a92668, 0x78200416, 0xe899b709, 0x189ebec3, 0x4b2a4be7, 0xe743447e, 0x49aba517, 0x1e96ed7d, 0x21498d64, 0x8659409b, 0x4f53ba5, 0x15685139, 0xadc82a53, 0xeab76ca6, 0x362d4288, 0xe9132520, 0x43fadda2, 0x4c86af0a, 0x7dd7bd17, 0x959a868f, 0xda92aef, 0x90df9c57, 0x92f95f19, 0x9d2b0fc7, 0x259000f3, 0xf4df855c, 0x96dcba4a, 0x42cf84be, 0xb3235d2e, 0x3b23d3a0, 0x1a3296d8, 0x88901497, 0x69ff92b7, 0xf32f2541, 0xb1c43571, 0x7e39274d, 0x6a07a703, 0x8a64cb80, 0xcc9bff8, 0xc3f6d4fd, 0x893081cd, 0xb9bc706, 0xa9ee7234, 0x23d4d1d5, 0x83cfc5c6, 0xcadaf602, 0xf1d0f64c, 0x7a10fdb, 0x96287895, 0xe21482cb, 0xe7455907, 0x8a71184d, 0x23318ac8, 0x72aed44b, 0x23a79d4c, 0xe463042b, 0xceb13214, 0x60a95e1e, 0x65ce2624, 0x89d63367, 0x2a54a707, 0xa6fd0f2d, 0xed7133c4, 0x5c7b9a39, 0x59072ef0, 0x922d9dfc, 0xc03538d2, 0xa3494060, 0xb9db010d, 0x20562c0a, 0xe2205917, 0xb0b13f7c, 0x9fa8fb51, 0xa5d2e787, 0xc03dc379, 0x5a9bc098, 0x595166ca, 0xc3633ddc, 0x4519fe94, 0x5b45a9a0, 0xe558cc8c, 0xc0b6bcc1, 0x6d97625e, 0xe19f9e3c, 0xf6ffc1e9, 0x2fcc098, 0xa9d74d52, 0x7f841373, 0x9f277361, 0x12e40882, 0x1d5ffdec, 0xc975a6dc, 0xf311ba67, 0x1066b762, 0x732d710a, 0x5af59eb6, 0x94115760, 0xf11c39e9, 0x687ba752, 0x613ac9d6, 0x3cadccc6, 0xc34ebac2, 0xe743e147, 0xc5873fe7, 0x92dca8b0, 0x4d25f636, 0xbc1c0f, 0xc5d23b85, 0x9dfd3591, 0x272668aa, 0x539f0014, 0x94215a76, 0x86ed47b0, 0x24e5916, 0xe2cd19e5, 0xb583cf8d, 0x5b7607f1, 0x787bea72, 0xe895c261, 0xc3bc56b5, 0x9f948754, 0x77a70c0a, 0x40cdc60, 0x44dd40b7, 0xedef993d, 0xd4e9ce38, 0xb0de9ed9, 0x8d23a195, 0xff508ce7, 0x11f0dc60, 0x2c1e81f9, 0x78f41fd3, 0x231b5419, 0xbe25d5f1, 0xeebe938e, 0x3124088d, 0xb262acd4, 0x76dd1e74, 0x10e5632b, 0x3af90e86, 0xc16606cd, 0x821c82b8, 0xc0fe07bb, 0x3564bb8f, 0xec4b5618, 0xc9f33fb, 0xb6267c89, 0x84d421c2, 0x1fd038c7, 0x2c1a6f35, 0x516dd45, 0x6574c9a4, 0x6b5c365, 0x1b7c4af8, 0x73f1815, 0x8282ed28, 0x3f07b04d, 0x1a2588db, 0xdc2acab1, 0x49e96971, 0x89f0d52f, 0x3b5485cd, 0x8d85bda8, 0xf7b915ce, 0xd78f562d, 0x473fe757, 0x1fc9d290, 0x3bceaf9c, 0x477894ea, 0x17703e2d, 0x95f451c3, 0x91efe374, 0xf8376d42, 0x6afd3bce, 0x8f9fad2e, 0x5e2af980, 0x4c6119d, 0xd020703f, 0xccf6434b, 0x4a2db623, 0x3ba433b5, 0xb7a6f9dd, 0xce9e7221, 0x9cea2a82, 0x634ad5d4, 0x6d49f653, 0xdd111430, 0xc07abf36, 0xbf435d20, 0x6d8520fd, 0x8e65c29c, 0x6daa7837, 0x22eebad2, 0x5ca5c51f, 0xf56c41c, 0x274130de, 0x1f105d4a, 0xefebb488, 0xb67c06d, 0xea99d663, 0x1b84bc20, 0xc82e6e08, 0x39c81654, 0x9dc7bbcd, 0xb6867232, 0x937424f, 0x8f499ca9, 0xa781c340, 0x336de6c, 0x213894d3, 0x126ce818, 0x4b5b2471, 0xf79f6bea, 0x5c5e5b14, 0x916f082c, 0x464f250f, 0x340b069e, 0x12efa1db, 0x40986b8e, 0x26f275f3, 0xbe6b5f13, 0x295fd4fb, 0x3207aa9a, 0xf04f68eb, 0x94d3f056, 0x5f02a187, 0xf8533ae3, 0xccdfc35, 0xd8c9c5af, 0xbdaf436e, 0x3d66a9bc, 0x4c764ef3, 0x7e67b122, 0x3450d064, 0x9eb56ae4, 0xfb546720, 0xd4381793, 0xe49d09c3, 0x28237b18, 0xc264aed3, 0x4c168e5, 0xa6afd65e, 0x11ecf8a6, 0xa28b2af, 0x7c811ce7, 0xd05a3385, 0x9b38684f, 0xc337fe3f, 0x91a03f71, 0x748d6457, 0x513ea9a0, 0x8825c72a, 0xfd15f2ca, 0x1bfd72c2, 0x946a2ed8, 0x256266e2, 0x6150ee9e, 0x823ffa91, 0x8d0af144, 0x3734db40, 0xbece3cbc, 0xea9c0df4, 0xab52d8ce, 0x764fa091, 0x43d1c4a7, 0x21dc44a3, 0x10fb51d1, 0xc3a5569c, 0x5ebdd696, 0x3675b726, 0xa1352aa9, 0x130299ca, 0x35cdf6cd, 0x186d6546, 0xc0b49699, 0x29864ce6, 0x110888f0, 0x5b413430, 0x42a402b2, 0x186617be, 0xce04652c, 0x24ace269, 0x3184e8b6, 0xee56dd00, 0xa3b3b556, 0x939b05f7, 0x8650cb10, 0xd6f2fd13, 0xecc03709, 0x9f901270, 0x43676adc, 0x53139433, 0x10e2d36f, 0xba5ef88a, 0xbc2d487b, 0x85d21144, 0xc5ad7597, 0xbc9fad0e, 0xe8669ebf, 0x2a580e15, 0xee95b8ae, 0xafe30d7c, 0x3925483, 0x6d4379b8, 0x3bff2758, 0xba82bcd0, 0x9de52fae, 0xc19f3236, 0xf2f2bbcf, 0x65c9da47, 0xf33ebdfc, 0xecd9ea0e, 0x8e80d2cd, 0x3419408f, 0xea652d99, 0x20c7735, 0xa43daed4, 0xed2551f9, 0x63d78bba, 0x4b651bd5, 0x7c69e272, 0x2abd56ee, 0x9ab4179d, 0x62b862c, 0x8caf2c25, 0x73b66096, 0xed1a39db, 0xceef49b6, 0xeb321d4c, 0x5e4df20c, 0x65169178, 0x9bf4fd3b, 0xf6985f9a, 0x8ebc7cc, 0x2c885bc7, 0x5ac592d7, 0x205721a7, 0x1419e27d, 0x22948c2b, 0xb17495cc, 0x81731372, 0x3412326, 0x582a65b, 0x75a4c270, 0xf2ae2cfd, 0xf618e345, 0xd3bdd6e2, 0x558d29a1, 0x3d848ab, 0x78da759a, 0x2d1b32b8, 0x1afaebc5, 0x55021e49, 0x80c99534, 0x21890259, 0xe2bc921b, 0xcf3ac82b, 0x88ccf401, 0x5840a952, 0x480edc7f, 0xf0aae25e, 0x5ac60fe0, 0x94fee19a, 0xe5729f54, 0xe0fb221e, 0x3dd23b67, 0xd8408440, 0x61d4496, 0xe7c7ca62, 0xf739df34, 0x75bab683, 0x6e126046, 0x8bdd628c, 0x58a65d27, 0xcc7148e6, 0x93bab453, 0x4924deef, 0x206404b0, 0x7d81413d, 0xf33609c6, 0x995f4b0d, 0x5048cca4, 0x3f9fd17, 0xf3ecbd96, 0x97ee65c2, 0x37e89e7a, 0x6f0569dc, 0x3faad0f9, 0xceaf2dea, 0xdd221d5f, 0x50b3e235, 0x3c3a8b89, 0xe4945aa9, 0xd0aa65b9, 0x93ee7358, 0x8b3cd96d, 0x2f1b4af6, 0x30f66edd, 0xc9b5baf4, 0x96be36cc, 0x9cae1352, 0xc52b16a, 0xdccff8b, 0x4104069, 0x6b91d024, 0xcb9dc70, 0xadd77490, 0x6640149e, 0xeb297c73, 0x93affa6a, 0x1a58d4, 0xde159fb9, 0xfa0ce323, 0xc941eda9, 0x60679151, 0x41343302, 0xf949e0d5, 0x150ea85d, 0x9aceaca4, 0x2862b26d, 0xd42f3d2e, 0x35a15350, 0x93204b1b, 0x6ad99948, 0xa0c99a1c, 0x59023682, 0x491b86e3, 0xb32b998d, 0x963ab5a9, 0xb28ffb79, 0xc0016d24, 0x13211ae4, 0xdbba74a8, 0x9561412, 0xc14d10e1, 0xca161ba6, 0xb2b3e13f, 0xe68909f3, 0xdd4e862e, 0x1319d68, 0x529b38d9, 0x43a2d3e7, 0xabb8d070, 0x77a609ff, 0x736c7965, 0x5c1cea02, 0x61d17c62, 0x3d7ee71c, 0x692aeece, 0x5ee62cb, 0x66c75dba, 0x881b66aa, 0x5140131a, 0x215470e8, 0x9f361b95, 0x8fbbc63a, 0x6e24af7f, 0x2af6f590, 0xf94b17dc, 0x6bcb8e25, 0xad845be9, 0xaf47b4ed, 0x32d579e8, 0x461e54ea, 0x6d3c470b, 0x28f5796, 0x57e58a06, 0x38492860, 0xcc320e8e, 0x82e4864, 0xe1479157, 0xef53c8cb, 0xe7622fa3, 0xc8372ebf, 0xa9a8017b, 0x84810a20, 0x452c5e6e, 0x436e75b, 0x409b1ab1, 0x4c7e2be6, 0xdada977a, 0x8a6c9880, 0x87184e96, 0xdc215461, 0xcd5a6dc6, 0x2f63a6a4, 0x928e9b92, 0x15634b1b, 0xbbaf3a9c, 0xb556de5, 0x84ddf234, 0x719d6666, 0xc558a5f2, 0x4f212de, 0x91a0ad79, 0x11290332, 0x7738c32e, 0xcfd04207, 0x57ba75d5, 0xa2e3f110, 0x11762ae7, 0xc5d9869d, 0x60bfa38b, 0xb4a795bd, 0x14625ffb, 0x3cfb63fa, 0xfb99e70e, 0xb5ef968, 0x2e7e2e1e, 0x47fdd488, 0xcfd5f8a8, 0xdd7683b0, 0xdffd7c9e, 0x12b83597, 0xb03bd9c2, 0xa90b82dc, 0x91ca62c0, 0x2ff80ae7, 0x29356c8c, 0x85b49291, 0x778572fa, 0xe764a1a1, 0x585d320a, 0x79a39c1c, 0x399c37e9, 0xd8d673cb, 0x97aedb85, 0xd6ba3f51, 0x4ff0ea85, 0x23a7db79, 0xea93002a, 0x8a516656, 0xe8dddfb4, 0x7792e739, 0x41d28138, 0x37fb9a4e, 0x1c635f57, 0x70da1ffa, 0x31660175, 0x7af382d1, 0x7fe4bf8a, 0x55521ec8, 0xba849730, 0x24819f5, 0x354c3b13, 0x8166158c, 0x3f7f2099, 0xac53614d, 0xda072b10, 0xe329504, 0x6a750fd8, 0x28572376, 0x9ddf129b, 0x9ae80bb2, 0x3bd18b44, 0x51b8a683, 0x1a1a86ba, 0xe3f8ee0c, 0x840f3482, 0xca55eb4c, 0x7a25e960, 0x37ae0760, 0x27157d3d, 0x3340e58f, 0x9f2eb164, 0xae3151fa, 0x8b443f4a, 0x27fdd38c, 0xa7758b92, 0x82a7049e, 0x2500efbb, 0xfcf1ce85, 0xc0642090, 0x250a875f, 0x38d83660, 0xc5d63bc, 0x84f43dab, 0xe6290eec, 0xc909291f, 0xb71fd6eb, 0x5b77263, 0x4f71a52c, 0x53096b18, 0x6497bbbe, 0xdf77d74f, 0x14ae3e15, 0xd83ee393, 0x5a249f4d, 0x89d73e21, 0x426804, 0xddda0a27, 0x9fe7c87c, 0xf3267c43, 0x3344fca2, 0xd38f68a2, 0xa007502a, 0xdaa74f4e, 0x524aaad2, 0x19473f66, 0xc1d0c26, 0xa6bbe17a, 0xb92e6eb0, 0xb419b07a, 0xc25e56d1, 0x9c38bdfe, 0x68038f05, 0xccb395fa, 0xb44f4ce6, 0x8d9a8db, 0x95da19f5, 0xc52e5d88, 0xf8356128, 0xbb5290c5, 0x4dcafa05, 0x427b9807, 0x505245ce, 0x41cf4b25, 0x39f96369, 0xa1dea14e, 0xac1e1e21, 0x58656a1b, 0xc76772b3, 0xcbed3c4d, 0x833dcc1e, 0x7236a3ab, 0x9d8200ae, 0xc8624409, 0xbb07343, 0xfd8f8f03, 0x723393c3, 0x4cdd1336, 0xf7cfad61, 0x249c07bf, 0x5c0c24e3, 0xe6bc2650, 0x8dd50d1c, 0x8aa3a2a5, 0x694e85c0, 0xf98895a5, 0xf74ed3dd, 0xa2f88192, 0x33fb994e, 0xfe711567, 0xfe29e1c6, 0x8bcb4bdc, 0xa8c15434, 0x86c3d98a, 0xcf1fafa0, 0x22ab672e, 0x7549603f, 0x5b0f7e7b, 0xe3269edd, 0x6b64292, 0x88afeba4, 0x290fa89d, 0x93c895f8, 0xbee21427, 0xeae12b1c, 0x7c84636, 0x506be81d, 0x5dd5c4b7, 0xb9e43567, 0xdcc2b95a, 0xce23ef9b, 0xb15353bc, 0x7ae5676a, 0xb0e1958c, 0x7cd1ac91, 0x304a8047, 0xd766a65b, 0x7120a387, 0x8e370d7c, 0x94e241d3, 0x4516f0b8, 0xfd605e33, 0x7c9b2ba0, 0x343332cc, 0x469dd63e, 0x3f6b6c66, 0x39b330b3, 0x431dc660, 0xa591aa1e, 0xc00b4c96, 0xebcf6905, 0x74fc55db, 0x3352505f, 0xe92ed69, 0xe13e0d96, 0x822e1ecc, 0x773f2c74, 0x3643056d, 0xb9984c16, 0xcc7153e6, 0x4d88f992, 0x4c1de8dd, 0x7d2f951d, 0x7113012, 0x34240034, 0x97eb307a, 0x16af8e1f, 0xd805ed31, 0x2a6aa2e2, 0x618a45d3, 0x80c46179, 0xbff6b6dd, 0xc99ee8ad, 0x82da2aeb, 0xe75273b3, 0x8a7bd156, 0x3832bb0f, 0xf59d7cc7, 0xbcf626b0, 0xcdd0619c, 0x41ff5ed3, 0x846531b, 0x906ffbfd, 0xb59a672b, 0x21e6e893, 0x770a564f, 0x5f412346, 0xf2929158, 0x153603fa, 0x38afd0d6, 0x8177460b, 0x445ed4e5, 0xb1c4a5c1, 0x14db7a5d, 0x74fc810f, 0x6db9f6b3, 0x5c7c7fe2, 0x1be8a756, 0xd0e5de2d, 0xa23fd973, 0xbba4dbf, 0xcd8effaa, 0x6348ae96, 0xb261846d, 0x7561c667, 0xc4266dd2, 0xbc58f0a8, 0x57ab12ac, 0xe8ddf790, 0xd889b911, 0x47040b6b, 0x6dc3ca58, 0xd15944e6, 0xd2f24f80, 0xa80450f0, 0xa0620d61, 0x2e34452, 0x24b75f77, 0xbc4023c7, 0x141129f4, 0x9a7ea00e, 0x4b172a4, 0xae15bc1a, 0x1114e3a0, 0x957e6381, 0x7565bfb0, 0x92d24558, 0x1d03f9f5, 0x5c6ba416, 0x71ccf22, 0xf56d5b5e, 0xc13e9a15, 0xfca848c4, 0x6514ebc3, 0x631b4eb0, 0xbf375c76, 0xd60fe492, 0x73d0cd61, 0xfb6dae8b, 0x7338e28b, 0x95d9b179, 0x7a62f5b4, 0xabc2334c, 0x79585244, 0x1614e8ad, 0xcd9c1cc0, 0xfdcfb459, 0x67033228, 0xc78d5307, 0xe7998307, 0xfe8eb0f6, 0x97cd521, 0xa758363a, 0xc61ca107, 0x994e0a2, 0x202a0884, 0xb8163d85, 0x9e56e7bd, 0xb0286076, 0x2a73b67, 0x9bdb2cb2, 0x89e37a4c, 0x1b30a9ad, 0xc5c8fe, 0x47daf560, 0xf3816f95, 0xc6b60072, 0xe7c86d7d, 0xe3e1f05c, 0xcbc47d47, 0x82b6a2dc, 0xea4f6aef, 0xb738a299, 0x25434de9, 0x13e950f7, 0x2860ff61, 0x9d849401, 0x3008a10d, 0xd67a5074, 0x9f5be426, 0xb22d68d5, 0xe7e3219b, 0x8e185ea6, 0xfd6db3d4, 0xdc76eca1, 0xb60c1f01, 0xbecd63fe, 0xbb57459f, 0x43b980f5, 0xe8c77071, 0xfff0de72, 0x669fa926, 0x92e0181c, 0x3ff636cb, 0xa3995e72, 0x2c65eb63, 0x7d5cf41f, 0x1e94bcb3, 0xe3ae1042, 0xd0073f43, 0xdfdfe7a0, 0x25942f4a, 0xd908c3a0, 0x43a8ffe1, 0xa5d13b05, 0xd1afd400, 0x95e87ba5, 0x4f82a56e, 0x4815cf48, 0xfb7fb5ab, 0xfddd096c, 0x44426f18, 0xccb69890, 0x889d817d, 0x806b3b60, 0x5080e8e0, 0xc5b67529, 0xe9286898, 0x1fd0d115, 0x5dd6a7f9, 0xebc7d39b, 0x6effe82e, 0xb7a6923, 0x83259d3e, 0x3a85d844, 0xf05add3e, 0xd7ad2f7d, 0x7ec4620, 0x4e39f0f9, 0xb7855c2d, 0x7bfc7f58, 0xe419d166, 0xa669e973, 0x6fc4baa, 0x24d97039, 0x85a52699, 0x4e48cb20, 0x53741168, 0xe30d3821, 0xdc077df6, 0xe3081286, 0x8efaf0c1, 0x83e53d33, 0xb0b2c61f, 0x445fd987, 0x73ee34fb, 0xe15e308d, 0xa0d8dc7f, 0x7f7bd583, 0x4a43c99c, 0x555c468f, 0x264ab1d, 0xe50ad642, 0x93a646ef, 0x3459003d, 0x4fbaced3, 0xd1d19c60, 0x846ba63a, 0xd6be13d2, 0xea9992e8, 0x85d5f5fa, 0x6d2d73a2, 0x2974c31e, 0x3f5558b9, 0xf2f02860, 0x5f0d1bb1, 0xf380dfc, 0xee8e732e, 0xe2f1a153, 0xefd69b29, 0x93083914, 0xd82601f8, 0x60fc1c28, 0xeb92a778, 0xa4125dfe, 0x3a57ac6e, 0x20b7f5b2], 3);
    test(0x1, [0x77eff, 0x20ccc389, 0x4d65aacb, 0xffc11e85, 0x2591cb4f, 0x3c7053c0, 0x17a38090, 0x65865081, 0x2faebfcc, 0x634e1e47, 0x5876aaed, 0xab7479fc, 0x65928d86, 0xef7f7d19, 0x89efe4b2, 0xd8a7d514, 0x6b5054fe, 0x5032b165, 0xaf6a8b92, 0x8648c9d4, 0x3456fb8b, 0x718620fc, 0xe0cc4dea, 0x3ac5929c, 0x702df9d, 0x88cf598e, 0xaba3c232, 0xe9f96a44, 0x6ad47cd7, 0x750b5fd7, 0x8f064be9, 0x6e4242f1, 0x23f065f7, 0xf06aadde, 0x32b6c760, 0xc7448456, 0xccfd988d, 0xb749d7e9, 0xf7dffd89, 0xcd818407, 0x503c913f, 0x17c1cb3d, 0xb13c0e16, 0x84a5a539, 0xe05b0a01, 0xdd71f780, 0xe504f13c, 0xd442f8d1, 0x15c57508, 0xd460e2d8, 0x9ff7e8d, 0x45e69a73, 0x2b7a01af, 0xf2a8fea, 0xe0cdf279, 0xaba7b9c8, 0x192d421e, 0x97d32447, 0x6bcdb583, 0xabf185c6, 0xf5383f95, 0x696aee31, 0x887d8539, 0x3292b203, 0xb11edbc4, 0x4a252b18, 0x50c5a1f1, 0x2461fc2b, 0xafbe864c, 0xc8874c1a, 0xd5aa0734, 0x699c2550, 0x4ae8a6d, 0x8bf7373, 0xc0097529, 0x9fc06dd2, 0xfd26000f, 0xa91f6c40, 0xbf87c8c4, 0x4c6a3019, 0x47cb2d6f, 0x7235dee2, 0xca0e35d7, 0x38dd2787, 0x1a6d051a, 0x12c7fe99, 0x72a92668, 0x78200416, 0xe899b709, 0x189ebec3, 0x4b2a4be7, 0xe743447e, 0x49aba517, 0x1e96ed7d, 0x21498d64, 0x8659409b, 0x4f53ba5, 0x15685139, 0xadc82a53, 0xeab76ca6, 0x362d4288, 0xe9132520, 0x43fadda2, 0x4c86af0a, 0x7dd7bd17, 0x959a868f, 0xda92aef, 0x90df9c57, 0x92f95f19, 0x9d2b0fc7, 0x259000f3, 0xf4df855c, 0x96dcba4a, 0x42cf84be, 0xb3235d2e, 0x3b23d3a0, 0x1a3296d8, 0x88901497, 0x69ff92b7, 0xf32f2541, 0xb1c43571, 0x7e39274d, 0x6a07a703, 0x8a64cb80, 0xcc9bff8, 0xc3f6d4fd, 0x893081cd, 0xb9bc706, 0xa9ee7234, 0x23d4d1d5, 0x83cfc5c6, 0xcadaf602, 0xf1d0f64c, 0x7a10fdb, 0x96287895, 0xe21482cb, 0xe7455907, 0x8a71184d, 0x23318ac8, 0x72aed44b, 0x23a79d4c, 0xe463042b, 0xceb13214, 0x60a95e1e, 0x65ce2624, 0x89d63367, 0x2a54a707, 0xa6fd0f2d, 0xed7133c4, 0x5c7b9a39, 0x59072ef0, 0x922d9dfc, 0xc03538d2, 0xa3494060, 0xb9db010d, 0x20562c0a, 0xe2205917, 0xb0b13f7c, 0x9fa8fb51, 0xa5d2e787, 0xc03dc379, 0x5a9bc098, 0x595166ca, 0xc3633ddc, 0x4519fe94, 0x5b45a9a0, 0xe558cc8c, 0xc0b6bcc1, 0x6d97625e, 0xe19f9e3c, 0xf6ffc1e9, 0x2fcc098, 0xa9d74d52, 0x7f841373, 0x9f277361, 0x12e40882, 0x1d5ffdec, 0xc975a6dc, 0xf311ba67, 0x1066b762, 0x732d710a, 0x5af59eb6, 0x94115760, 0xf11c39e9, 0x687ba752, 0x613ac9d6, 0x3cadccc6, 0xc34ebac2, 0xe743e147, 0xc5873fe7, 0x92dca8b0, 0x4d25f636, 0xbc1c0f, 0xc5d23b85, 0x9dfd3591, 0x272668aa, 0x539f0014, 0x94215a76, 0x86ed47b0, 0x24e5916, 0xe2cd19e5, 0xb583cf8d, 0x5b7607f1, 0x787bea72, 0xe895c261, 0xc3bc56b5, 0x9f948754, 0x77a70c0a, 0x40cdc60, 0x44dd40b7, 0xedef993d, 0xd4e9ce38, 0xb0de9ed9, 0x8d23a195, 0xff508ce7, 0x11f0dc60, 0x2c1e81f9, 0x78f41fd3, 0x231b5419, 0xbe25d5f1, 0xeebe938e, 0x3124088d, 0xb262acd4, 0x76dd1e74, 0x10e5632b, 0x3af90e86, 0xc16606cd, 0x821c82b8, 0xc0fe07bb, 0x3564bb8f, 0xec4b5618, 0xc9f33fb, 0xb6267c89, 0x84d421c2, 0x1fd038c7, 0x2c1a6f35, 0x516dd45, 0x6574c9a4, 0x6b5c365, 0x1b7c4af8, 0x73f1815, 0x8282ed28, 0x3f07b04d, 0x1a2588db, 0xdc2acab1, 0x49e96971, 0x89f0d52f, 0x3b5485cd, 0x8d85bda8, 0xf7b915ce, 0xd78f562d, 0x473fe757, 0x1fc9d290, 0x3bceaf9c, 0x477894ea, 0x17703e2d, 0x95f451c3, 0x91efe374, 0xf8376d42, 0x6afd3bce, 0x8f9fad2e, 0x5e2af980, 0x4c6119d, 0xd020703f, 0xccf6434b, 0x4a2db623, 0x3ba433b5, 0xb7a6f9dd, 0xce9e7221, 0x9cea2a82, 0x634ad5d4, 0x6d49f653, 0xdd111430, 0xc07abf36, 0xbf435d20, 0x6d8520fd, 0x8e65c29c, 0x6daa7837, 0x22eebad2, 0x5ca5c51f, 0xf56c41c, 0x274130de, 0x1f105d4a, 0xefebb488, 0xb67c06d, 0xea99d663, 0x1b84bc20, 0xc82e6e08, 0x39c81654, 0x9dc7bbcd, 0xb6867232, 0x937424f, 0x8f499ca9, 0xa781c340, 0x336de6c, 0x213894d3, 0x126ce818, 0x4b5b2471, 0xf79f6bea, 0x5c5e5b14, 0x916f082c, 0x464f250f, 0x340b069e, 0x12efa1db, 0x40986b8e, 0x26f275f3, 0xbe6b5f13, 0x295fd4fb, 0x3207aa9a, 0xf04f68eb, 0x94d3f056, 0x5f02a187, 0xf8533ae3, 0xccdfc35, 0xd8c9c5af, 0xbdaf436e, 0x3d66a9bc, 0x4c764ef3, 0x7e67b122, 0x3450d064, 0x9eb56ae4, 0xfb546720, 0xd4381793, 0xe49d09c3, 0x28237b18, 0xc264aed3, 0x4c168e5, 0xa6afd65e, 0x11ecf8a6, 0xa28b2af, 0x7c811ce7, 0xd05a3385, 0x9b38684f, 0xc337fe3f, 0x91a03f71, 0x748d6457, 0x513ea9a0, 0x8825c72a, 0xfd15f2ca, 0x1bfd72c2, 0x946a2ed8, 0x256266e2, 0x6150ee9e, 0x823ffa91, 0x8d0af144, 0x3734db40, 0xbece3cbc, 0xea9c0df4, 0xab52d8ce, 0x764fa091, 0x43d1c4a7, 0x21dc44a3, 0x10fb51d1, 0xc3a5569c, 0x5ebdd696, 0x3675b726, 0xa1352aa9, 0x130299ca, 0x35cdf6cd, 0x186d6546, 0xc0b49699, 0x29864ce6, 0x110888f0, 0x5b413430, 0x42a402b2, 0x186617be, 0xce04652c, 0x24ace269, 0x3184e8b6, 0xee56dd00, 0xa3b3b556, 0x939b05f7, 0x8650cb10, 0xd6f2fd13, 0xecc03709, 0x9f901270, 0x43676adc, 0x53139433, 0x10e2d36f, 0xba5ef88a, 0xbc2d487b, 0x85d21144, 0xc5ad7597, 0xbc9fad0e, 0xe8669ebf, 0x2a580e15, 0xee95b8ae, 0xafe30d7c, 0x3925483, 0x6d4379b8, 0x3bff2758, 0xba82bcd0, 0x9de52fae, 0xc19f3236, 0xf2f2bbcf, 0x65c9da47, 0xf33ebdfc, 0xecd9ea0e, 0x8e80d2cd, 0x3419408f, 0xea652d99, 0x20c7735, 0xa43daed4, 0xed2551f9, 0x63d78bba, 0x4b651bd5, 0x7c69e272, 0x2abd56ee, 0x9ab4179d, 0x62b862c, 0x8caf2c25, 0x73b66096, 0xed1a39db, 0xceef49b6, 0xeb321d4c, 0x5e4df20c, 0x65169178, 0x9bf4fd3b, 0xf6985f9a, 0x8ebc7cc, 0x2c885bc7, 0x5ac592d7, 0x205721a7, 0x1419e27d, 0x22948c2b, 0xb17495cc, 0x81731372, 0x3412326, 0x582a65b, 0x75a4c270, 0xf2ae2cfd, 0xf618e345, 0xd3bdd6e2, 0x558d29a1, 0x3d848ab, 0x78da759a, 0x2d1b32b8, 0x1afaebc5, 0x55021e49, 0x80c99534, 0x21890259, 0xe2bc921b, 0xcf3ac82b, 0x88ccf401, 0x5840a952, 0x480edc7f, 0xf0aae25e, 0x5ac60fe0, 0x94fee19a, 0xe5729f54, 0xe0fb221e, 0x3dd23b67, 0xd8408440, 0x61d4496, 0xe7c7ca62, 0xf739df34, 0x75bab683, 0x6e126046, 0x8bdd628c, 0x58a65d27, 0xcc7148e6, 0x93bab453, 0x4924deef, 0x206404b0, 0x7d81413d, 0xf33609c6, 0x995f4b0d, 0x5048cca4, 0x3f9fd17, 0xf3ecbd96, 0x97ee65c2, 0x37e89e7a, 0x6f0569dc, 0x3faad0f9, 0xceaf2dea, 0xdd221d5f, 0x50b3e235, 0x3c3a8b89, 0xe4945aa9, 0xd0aa65b9, 0x93ee7358, 0x8b3cd96d, 0x2f1b4af6, 0x30f66edd, 0xc9b5baf4, 0x96be36cc, 0x9cae1352, 0xc52b16a, 0xdccff8b, 0x4104069, 0x6b91d024, 0xcb9dc70, 0xadd77490, 0x6640149e, 0xeb297c73, 0x93affa6a, 0x1a58d4, 0xde159fb9, 0xfa0ce323, 0xc941eda9, 0x60679151, 0x41343302, 0xf949e0d5, 0x150ea85d, 0x9aceaca4, 0x2862b26d, 0xd42f3d2e, 0x35a15350, 0x93204b1b, 0x6ad99948, 0xa0c99a1c, 0x59023682, 0x491b86e3, 0xb32b998d, 0x963ab5a9, 0xb28ffb79, 0xc0016d24, 0x13211ae4, 0xdbba74a8, 0x9561412, 0xc14d10e1, 0xca161ba6, 0xb2b3e13f, 0xe68909f3, 0xdd4e862e, 0x1319d68, 0x529b38d9, 0x43a2d3e7, 0xabb8d070, 0x77a609ff, 0x736c7965, 0x5c1cea02, 0x61d17c62, 0x3d7ee71c, 0x692aeece, 0x5ee62cb, 0x66c75dba, 0x881b66aa, 0x5140131a, 0x215470e8, 0x9f361b95, 0x8fbbc63a, 0x6e24af7f, 0x2af6f590, 0xf94b17dc, 0x6bcb8e25, 0xad845be9, 0xaf47b4ed, 0x32d579e8, 0x461e54ea, 0x6d3c470b, 0x28f5796, 0x57e58a06, 0x38492860, 0xcc320e8e, 0x82e4864, 0xe1479157, 0xef53c8cb, 0xe7622fa3, 0xc8372ebf, 0xa9a8017b, 0x84810a20, 0x452c5e6e, 0x436e75b, 0x409b1ab1, 0x4c7e2be6, 0xdada977a, 0x8a6c9880, 0x87184e96, 0xdc215461, 0xcd5a6dc6, 0x2f63a6a4, 0x928e9b92, 0x15634b1b, 0xbbaf3a9c, 0xb556de5, 0x84ddf234, 0x719d6666, 0xc558a5f2, 0x4f212de, 0x91a0ad79, 0x11290332, 0x7738c32e, 0xcfd04207, 0x57ba75d5, 0xa2e3f110, 0x11762ae7, 0xc5d9869d, 0x60bfa38b, 0xb4a795bd, 0x14625ffb, 0x3cfb63fa, 0xfb99e70e, 0xb5ef968, 0x2e7e2e1e, 0x47fdd488, 0xcfd5f8a8, 0xdd7683b0, 0xdffd7c9e, 0x12b83597, 0xb03bd9c2, 0xa90b82dc, 0x91ca62c0, 0x2ff80ae7, 0x29356c8c, 0x85b49291, 0x778572fa, 0xe764a1a1, 0x585d320a, 0x79a39c1c, 0x399c37e9, 0xd8d673cb, 0x97aedb85, 0xd6ba3f51, 0x4ff0ea85, 0x23a7db79, 0xea93002a, 0x8a516656, 0xe8dddfb4, 0x7792e739, 0x41d28138, 0x37fb9a4e, 0x1c635f57, 0x70da1ffa, 0x31660175, 0x7af382d1, 0x7fe4bf8a, 0x55521ec8, 0xba849730, 0x24819f5, 0x354c3b13, 0x8166158c, 0x3f7f2099, 0xac53614d, 0xda072b10, 0xe329504, 0x6a750fd8, 0x28572376, 0x9ddf129b, 0x9ae80bb2, 0x3bd18b44, 0x51b8a683, 0x1a1a86ba, 0xe3f8ee0c, 0x840f3482, 0xca55eb4c, 0x7a25e960, 0x37ae0760, 0x27157d3d, 0x3340e58f, 0x9f2eb164, 0xae3151fa, 0x8b443f4a, 0x27fdd38c, 0xa7758b92, 0x82a7049e, 0x2500efbb, 0xfcf1ce85, 0xc0642090, 0x250a875f, 0x38d83660, 0xc5d63bc, 0x84f43dab, 0xe6290eec, 0xc909291f, 0xb71fd6eb, 0x5b77263, 0x4f71a52c, 0x53096b18, 0x6497bbbe, 0xdf77d74f, 0x14ae3e15, 0xd83ee393, 0x5a249f4d, 0x89d73e21, 0x426804, 0xddda0a27, 0x9fe7c87c, 0xf3267c43, 0x3344fca2, 0xd38f68a2, 0xa007502a, 0xdaa74f4e, 0x524aaad2, 0x19473f66, 0xc1d0c26, 0xa6bbe17a, 0xb92e6eb0, 0xb419b07a, 0xc25e56d1, 0x9c38bdfe, 0x68038f05, 0xccb395fa, 0xb44f4ce6, 0x8d9a8db, 0x95da19f5, 0xc52e5d88, 0xf8356128, 0xbb5290c5, 0x4dcafa05, 0x427b9807, 0x505245ce, 0x41cf4b25, 0x39f96369, 0xa1dea14e, 0xac1e1e21, 0x58656a1b, 0xc76772b3, 0xcbed3c4d, 0x833dcc1e, 0x7236a3ab, 0x9d8200ae, 0xc8624409, 0xbb07343, 0xfd8f8f03, 0x723393c3, 0x4cdd1336, 0xf7cfad61, 0x249c07bf, 0x5c0c24e3, 0xe6bc2650, 0x8dd50d1c, 0x8aa3a2a5, 0x694e85c0, 0xf98895a5, 0xf74ed3dd, 0xa2f88192, 0x33fb994e, 0xfe711567, 0xfe29e1c6, 0x8bcb4bdc, 0xa8c15434, 0x86c3d98a, 0xcf1fafa0, 0x22ab672e, 0x7549603f, 0x5b0f7e7b, 0xe3269edd, 0x6b64292, 0x88afeba4, 0x290fa89d, 0x93c895f8, 0xbee21427, 0xeae12b1c, 0x7c84636, 0x506be81d, 0x5dd5c4b7, 0xb9e43567, 0xdcc2b95a, 0xce23ef9b, 0xb15353bc, 0x7ae5676a, 0xb0e1958c, 0x7cd1ac91, 0x304a8047, 0xd766a65b, 0x7120a387, 0x8e370d7c, 0x94e241d3, 0x4516f0b8, 0xfd605e33, 0x7c9b2ba0, 0x343332cc, 0x469dd63e, 0x3f6b6c66, 0x39b330b3, 0x431dc660, 0xa591aa1e, 0xc00b4c96, 0xebcf6905, 0x74fc55db, 0x3352505f, 0xe92ed69, 0xe13e0d96, 0x822e1ecc, 0x773f2c74, 0x3643056d, 0xb9984c16, 0xcc7153e6, 0x4d88f992, 0x4c1de8dd, 0x7d2f951d, 0x7113012, 0x34240034, 0x97eb307a, 0x16af8e1f, 0xd805ed31, 0x2a6aa2e2, 0x618a45d3, 0x80c46179, 0xbff6b6dd, 0xc99ee8ad, 0x82da2aeb, 0xe75273b3, 0x8a7bd156, 0x3832bb0f, 0xf59d7cc7, 0xbcf626b0, 0xcdd0619c, 0x41ff5ed3, 0x846531b, 0x906ffbfd, 0xb59a672b, 0x21e6e893, 0x770a564f, 0x5f412346, 0xf2929158, 0x153603fa, 0x38afd0d6, 0x8177460b, 0x445ed4e5, 0xb1c4a5c1, 0x14db7a5d, 0x74fc810f, 0x6db9f6b3, 0x5c7c7fe2, 0x1be8a756, 0xd0e5de2d, 0xa23fd973, 0xbba4dbf, 0xcd8effaa, 0x6348ae96, 0xb261846d, 0x7561c667, 0xc4266dd2, 0xbc58f0a8, 0x57ab12ac, 0xe8ddf790, 0xd889b911, 0x47040b6b, 0x6dc3ca58, 0xd15944e6, 0xd2f24f80, 0xa80450f0, 0xa0620d61, 0x2e34452, 0x24b75f77, 0xbc4023c7, 0x141129f4, 0x9a7ea00e, 0x4b172a4, 0xae15bc1a, 0x1114e3a0, 0x957e6381, 0x7565bfb0, 0x92d24558, 0x1d03f9f5, 0x5c6ba416, 0x71ccf22, 0xf56d5b5e, 0xc13e9a15, 0xfca848c4, 0x6514ebc3, 0x631b4eb0, 0xbf375c76, 0xd60fe492, 0x73d0cd61, 0xfb6dae8b, 0x7338e28b, 0x95d9b179, 0x7a62f5b4, 0xabc2334c, 0x79585244, 0x1614e8ad, 0xcd9c1cc0, 0xfdcfb459, 0x67033228, 0xc78d5307, 0xe7998307, 0xfe8eb0f6, 0x97cd521, 0xa758363a, 0xc61ca107, 0x994e0a2, 0x202a0884, 0xb8163d85, 0x9e56e7bd, 0xb0286076, 0x2a73b67, 0x9bdb2cb2, 0x89e37a4c, 0x1b30a9ad, 0xc5c8fe, 0x47daf560, 0xf3816f95, 0xc6b60072, 0xe7c86d7d, 0xe3e1f05c, 0xcbc47d47, 0x82b6a2dc, 0xea4f6aef, 0xb738a299, 0x25434de9, 0x13e950f7, 0x2860ff61, 0x9d849401, 0x3008a10d, 0xd67a5074, 0x9f5be426, 0xb22d68d5, 0xe7e3219b, 0x8e185ea6, 0xfd6db3d4, 0xdc76eca1, 0xb60c1f01, 0xbecd63fe, 0xbb57459f, 0x43b980f5, 0xe8c77071, 0xfff0de72, 0x669fa926, 0x92e0181c, 0x3ff636cb, 0xa3995e72, 0x2c65eb63, 0x7d5cf41f, 0x1e94bcb3, 0xe3ae1042, 0xd0073f43, 0xdfdfe7a0, 0x25942f4a, 0xd908c3a0, 0x43a8ffe1, 0xa5d13b05, 0xd1afd400, 0x95e87ba5, 0x4f82a56e, 0x4815cf48, 0xfb7fb5ab, 0xfddd096c, 0x44426f18, 0xccb69890, 0x889d817d, 0x806b3b60, 0x5080e8e0, 0xc5b67529, 0xe9286898, 0x1fd0d115, 0x5dd6a7f9, 0xebc7d39b, 0x6effe82e, 0xb7a6923, 0x83259d3e, 0x3a85d844, 0xf05add3e, 0xd7ad2f7d, 0x7ec4620, 0x4e39f0f9, 0xb7855c2d, 0x7bfc7f58, 0xe419d166, 0xa669e973, 0x6fc4baa, 0x24d97039, 0x85a52699, 0x4e48cb20, 0x53741168, 0xe30d3821, 0xdc077df6, 0xe3081286, 0x8efaf0c1, 0x83e53d33, 0xb0b2c61f, 0x445fd987, 0x73ee34fb, 0xe15e308d, 0xa0d8dc7f, 0x7f7bd583, 0x4a43c99c, 0x555c468f, 0x264ab1d, 0xe50ad642, 0x93a646ef, 0x3459003d, 0x4fbaced3, 0xd1d19c60, 0x846ba63a, 0xd6be13d2, 0xea9992e8, 0x85d5f5fa, 0x6d2d73a2, 0x2974c31e, 0x3f5558b9, 0xf2f02860, 0x5f0d1bb1, 0xf380dfc, 0xee8e732e, 0xe2f1a153, 0xefd69b29, 0x93083914, 0xd82601f8, 0x60fc1c28, 0xeb92a778, 0xa4125dfe, 0x3a57ac6e, 0x20b7f5b2], 4);
  });
}(typeof module !== "undefined" ? require("../lib/random") : Random));